CREATE SCHEMA pubsub;

SET search_path = pubsub, public;

CREATE TABLE watermark(
       consumer_id text NOT NULL,
       branch_id bigint NOT NULL,
       txid bigint NOT NULL DEFAULT 0,
       commit_id bigint NOT NULL DEFAULT 0,
       PRIMARY KEY(consumer_id, branch_id));

CREATE TABLE out_of_order(
       consumer_id text NOT NULL,
       branch_id bigint NOT NULL,
       txid bigint NOT NULL,
       commit_id bigint NOT NULL);

CREATE INDEX out_of_order_consumer_id_branch_id_idx ON out_of_order
    USING btree(consumer_id, branch_id);

CREATE OR REPLACE FUNCTION lock_watermark(
       p_consumer_id text,
       p_branch_id bigint)
RETURNS SETOF watermark AS $$
BEGIN
    LOOP
        BEGIN
            RETURN QUERY SELECT * FROM pubsub.watermark
              WHERE consumer_id=p_consumer_id AND branch_id=p_branch_id
              FOR UPDATE NOWAIT;
            IF found THEN RETURN; END IF;
        EXCEPTION WHEN lock_not_available THEN RETURN;
        END;
        
        BEGIN
            RETURN QUERY INSERT INTO pubsub.watermark
              (consumer_id, branch_id)
              VALUES(p_consumer_id, p_branch_id)
              RETURNING *;
            RETURN;
        EXCEPTION WHEN unique_violation THEN
            -- loop once more
        END;
    END LOOP;
end;
$$ LANGUAGE plpgsql;

SET search_path = revision, public;

CREATE TABLE commit_queue(
       branch_id bigint NOT NULL,
       txid bigint NOT NULL DEFAULT txid_current(),
       commit_id bigint NOT NULL,

       PRIMARY KEY (branch_id, txid, commit_id),
       CONSTRAINT valid_commit_id CHECK (commit_id > 0)
);

BEGIN WORK;

CREATE OR REPLACE FUNCTION commit_trigger() RETURNS TRIGGER
       LANGUAGE plpgsql
       AS $$
DECLARE
        branch_id bigint;
BEGIN
        IF TG_OP = 'UPDATE' AND NEW.state = OLD.state THEN
           RETURN NULL;
        END IF;

        IF NEW.state = 'draft' THEN
            branch_id := 0;
        ELSIF NEW.stable_branch_id IS NULL THEN
            branch_id := (SELECT id FROM revision.branch WHERE type='approved');
        ELSE
            branch_id := NEW.stable_branch_id;
        END IF;
        
        INSERT INTO revision.commit_queue(branch_id, commit_id)
            VALUES(branch_id, NEW.id);
        
        RETURN NULL;
END;
$$;

LOCK TABLE commit IN EXCLUSIVE MODE;

CREATE TRIGGER commit_trigger
       AFTER INSERT OR UPDATE ON revision.commit
       FOR EACH ROW EXECUTE PROCEDURE commit_trigger();

INSERT INTO commit_queue(branch_id, commit_id)
       SELECT 0, id FROM commit WHERE trunk;
INSERT INTO commit_queue(branch_id, commit_id)
       SELECT (SELECT id FROM branch WHERE type='approved'), id
       FROM commit WHERE trunk AND state='approved';
INSERT INTO commit_queue(branch_id, commit_id)
       SELECT stable_branch_id, id FROM commit WHERE NOT trunk;

COMMIT WORK;

SET search_path = public;
