CREATE OR REPLACE FUNCTION vrevisions_stable.merge_approved_to_stable_branch(
    branch_id bigint)
    RETURNS void AS
$$
DECLARE
    rowcount bigint;
BEGIN
    SELECT COUNT(*) INTO rowcount
        FROM vrevisions_stable.branch_mask
        WHERE branches ? branch_id::text;
    IF rowcount > 0 THEN
        RAISE EXCEPTION 'Branch % already exists', branch_id::text;
    END IF;

    UPDATE vrevisions_stable.branch_mask
        SET branches = branches || hstore(branch_id::text,'1')
        WHERE branches ? '1';

    INSERT INTO vrevisions_stable.branch_mask(branches)
        VALUES (hstore(branch_id::text,'1'));

    INSERT INTO vrevisions_stable.branch_mask(branches)
        VALUES ('1=>1'::hstore);
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION vrevisions_stable.delete_branch(
    branch_id bigint)
    RETURNS void AS
$$
DECLARE
    mask_id bigint;
    r record;
BEGIN
    UPDATE vrevisions_stable.branch_mask
        SET branches = branches - branch_id::text
        WHERE branches ? branch_id::text;
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION vrevisions_stable.get_branch_mask_id(
    in_branches hstore)
    RETURNS bigint AS
$$
DECLARE
    mask_id bigint;
BEGIN
    SELECT branch_mask_id INTO mask_id
        FROM vrevisions_stable.branch_mask
        WHERE branches=in_branches
        ORDER BY 1
        LIMIT 1;
    IF NOT FOUND THEN
        INSERT INTO vrevisions_stable.branch_mask (branches)
            VALUES (in_branches)
            RETURNING branch_mask_id INTO mask_id;
    END IF;
    RETURN mask_id;
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION vrevisions_stable.insert_relation(
    in_branch_id bigint,
    in_id bigint,
    in_commit_id bigint,
    in_domain_attrs hstore,
    in_master_id bigint,
    in_slave_id bigint)
    RETURNS void AS
$$
DECLARE
    old_branches hstore;
    old_mask_id bigint;
    new_mask_id bigint;
BEGIN
    SELECT branch_mask_id, branches INTO old_mask_id, old_branches
        FROM vrevisions_stable.objects_r
        JOIN vrevisions_stable.branch_mask
        USING (branch_mask_id)
        WHERE id=in_id AND commit_id=in_commit_id;
    IF FOUND THEN
        new_mask_id := vrevisions_stable.get_branch_mask_id(old_branches || hstore(in_branch_id::text,'1'));
        UPDATE vrevisions_stable.objects_r
            SET branch_mask_id=new_mask_id
            WHERE id=in_id AND branch_mask_id=old_mask_id;
    ELSE
        new_mask_id := vrevisions_stable.get_branch_mask_id(hstore(in_branch_id::text,'1'));
        INSERT INTO vrevisions_stable.objects_r (branch_mask_id, id, commit_id, domain_attrs, master_id, slave_id)
            VALUES (new_mask_id, in_id, in_commit_id, in_domain_attrs, in_master_id, in_slave_id);
    END IF;
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION vrevisions_stable.delete_relation(
    in_branch_id bigint,
    in_id bigint)
    RETURNS void AS
$$
DECLARE
    old_branches hstore;
    new_branches hstore;
    old_mask_id bigint;
    new_mask_id bigint;
BEGIN
    SELECT branch_mask_id, branches INTO old_mask_id, old_branches
        FROM vrevisions_stable.objects_r
        JOIN vrevisions_stable.branch_mask
        USING (branch_mask_id)
        WHERE id=in_id AND branches ? in_branch_id::text;
    IF FOUND THEN
        new_branches := old_branches - in_branch_id::text;
        IF new_branches=''::hstore THEN
            DELETE FROM vrevisions_stable.objects_r
                WHERE id=in_id AND branch_mask_id=old_mask_id;
        ELSE
            new_mask_id := vrevisions_stable.get_branch_mask_id(new_branches);
            UPDATE vrevisions_stable.objects_r
            SET branch_mask_id=new_mask_id
            WHERE id=in_id AND branch_mask_id=old_mask_id;
        END IF;
    END IF;
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION vrevisions_stable.update_relation(
    in_branch_id bigint,
    in_id bigint,
    in_commit_id bigint,
    in_domain_attrs hstore,
    in_master_id bigint,
    in_slave_id bigint)
    RETURNS void AS
$$
BEGIN
    PERFORM vrevisions_stable.delete_relation(in_branch_id, in_id);
    PERFORM vrevisions_stable.insert_relation(in_branch_id, in_id, in_commit_id, in_domain_attrs, in_master_id, in_slave_id);
END;
$$ LANGUAGE plpgsql;

