SET search_path = social,public;

----------------------------------
-- FEEDBACK AOI FEED BASE TABLE --
----------------------------------

CREATE TABLE feedback_aoi_feed(
    aoi_id bigint NOT NULL,
    feedback_task_id bigint NOT NULL
);

CREATE FUNCTION feedback_aoi_feed_insert_trigger() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
    RAISE EXCEPTION 'Not insertable table feedback_aoi_feed';
END;
$$;

CREATE TRIGGER feedback_aoi_feed_insert_trigger BEFORE INSERT ON feedback_aoi_feed
    FOR EACH STATEMENT EXECUTE PROCEDURE feedback_aoi_feed_insert_trigger();

------------------------------------
-- FEEDBACK AOI FEED CHILD TABLES --
------------------------------------

CREATE TABLE feedback_aoi_feed_pending () INHERITS (feedback_aoi_feed);
CREATE TABLE feedback_aoi_feed_outgoing_opened () INHERITS (feedback_aoi_feed);
CREATE TABLE feedback_aoi_feed_outgoing_closed () INHERITS (feedback_aoi_feed);

-----------------------------
-- CONSTRAINTS AND INDEXES --
-----------------------------

ALTER TABLE feedback_aoi_feed
    ADD CONSTRAINT feedback_aoi_feed_pkey PRIMARY KEY(aoi_id, feedback_task_id);

ALTER TABLE feedback_aoi_feed_pending
    ADD CONSTRAINT feedback_aoi_feed_pending_pkey PRIMARY KEY(aoi_id, feedback_task_id);

ALTER TABLE feedback_aoi_feed_outgoing_opened
    ADD CONSTRAINT feedback_aoi_feed_outgoing_opened_pkey PRIMARY KEY(aoi_id, feedback_task_id);

ALTER TABLE feedback_aoi_feed_outgoing_closed
    ADD CONSTRAINT feedback_aoi_feed_outgoing_closed_pkey PRIMARY KEY(aoi_id, feedback_task_id);

CREATE INDEX feedback_aoi_feed_feedback_id_idx ON feedback_aoi_feed USING btree (feedback_task_id);
CREATE INDEX feedback_aoi_feed_pending_feedback_id_idx ON feedback_aoi_feed_pending USING btree (feedback_task_id);
CREATE INDEX feedback_aoi_feed_outgoing_opened_feedback_id_idx ON feedback_aoi_feed_outgoing_opened USING btree (feedback_task_id);
CREATE INDEX feedback_aoi_feed_outgoing_closed_feedback_id_idx ON feedback_aoi_feed_outgoing_closed USING btree (feedback_task_id);

-------------------------------------------------
-- TRIGGERS FOR FEED TRANSITION BETWEEN TABLES --
-------------------------------------------------

-- 'pending' -> 'outgoing_opened', 'pending' -> 'outgoing_closed'
--
CREATE OR REPLACE FUNCTION feedback_task_pending_update_trigger() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
    IF (NEW.bucket != 'outgoing') THEN
        RETURN NEW;
    END IF;

    IF (NEW.closed_by IS NULL) THEN

        INSERT INTO social.feedback_task_outgoing_opened VALUES (NEW.*);

        -- update outgoing opened feed
        --
        INSERT INTO social.feedback_aoi_feed_outgoing_opened
            SELECT * FROM social.feedback_aoi_feed_pending WHERE feedback_task_id = NEW.id;

    ELSE

        INSERT INTO social.feedback_task_outgoing_closed VALUES (NEW.*);

        -- update outgoing closed feed
        --
        INSERT INTO social.feedback_aoi_feed_outgoing_closed
            SELECT * FROM social.feedback_aoi_feed_pending WHERE feedback_task_id = NEW.id;

    END IF;

    DELETE FROM social.feedback_task_pending WHERE id = NEW.id;
    DELETE FROM social.feedback_aoi_feed_pending WHERE feedback_task_id = NEW.id;

    RETURN NULL;
END;
$$;

-- 'outgoing_opened' -> 'outgoing_closed'
--
CREATE OR REPLACE FUNCTION feedback_task_outgoing_opened_update_trigger() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
    IF (NEW.closed_by IS NULL) THEN
        RETURN NEW;
    END IF;

    DELETE FROM social.feedback_task_outgoing_opened WHERE id = NEW.id;
    INSERT INTO social.feedback_task_outgoing_closed VALUES (NEW.*);

    -- update outgoing closed feed
    --
    INSERT INTO social.feedback_aoi_feed_outgoing_closed
        SELECT * FROM social.feedback_aoi_feed_outgoing_opened WHERE feedback_task_id = NEW.id;
    DELETE FROM social.feedback_aoi_feed_outgoing_opened WHERE feedback_task_id = NEW.id;

    RETURN NULL;
END;
$$;

-- 'outgoing_closed' -> 'outgoing_opened'
--
CREATE OR REPLACE FUNCTION feedback_task_outgoing_closed_update_trigger() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
    IF (NEW.closed_by IS NOT NULL) THEN
        RETURN NEW;
    END IF;

    DELETE FROM social.feedback_task_outgoing_closed WHERE id = NEW.id;
    INSERT INTO social.feedback_task_outgoing_opened VALUES (NEW.*);

    -- update outgoing opened feed
    --
    INSERT INTO social.feedback_aoi_feed_outgoing_opened
        SELECT * FROM social.feedback_aoi_feed_outgoing_closed WHERE feedback_task_id = NEW.id;
    DELETE FROM social.feedback_aoi_feed_outgoing_closed WHERE feedback_task_id = NEW.id;

    RETURN NULL;
END;
$$;


SET search_path = public;
