-- modify update triggers to react on resolved_by instead of closed_by
--
CREATE OR REPLACE FUNCTION social.feedback_task_outgoing_opened_update_trigger() RETURNS trigger
    LANGUAGE plpgsql
AS
$$
BEGIN
    IF (NEW.resolved_by IS NULL) THEN
        RETURN NEW;
    END IF;

    DELETE FROM social.feedback_task_outgoing_opened WHERE id = NEW.id;
    INSERT INTO social.feedback_task_outgoing_closed VALUES (NEW.*);

    -- update outgoing closed feed
    --
    INSERT INTO social.feedback_aoi_feed_outgoing_closed
        SELECT * FROM social.feedback_aoi_feed_outgoing_opened WHERE feedback_task_id = NEW.id;
    DELETE FROM social.feedback_aoi_feed_outgoing_opened WHERE feedback_task_id = NEW.id;

    RETURN NULL;
END;
$$;


CREATE OR REPLACE FUNCTION social.feedback_task_outgoing_closed_update_trigger() RETURNS trigger
    LANGUAGE plpgsql
AS
$$
BEGIN
    IF (NEW.resolved_by IS NOT NULL) THEN
        RETURN NEW;
    END IF;

    DELETE FROM social.feedback_task_outgoing_closed WHERE id = OLD.id;

    IF (NEW.type IN ('address-experiment', 'entrance-experiment')) THEN
        INSERT INTO social.feedback_task_outgoing_opened_experiment VALUES (NEW.*);
    ELSE
        INSERT INTO social.feedback_task_outgoing_opened VALUES (NEW.*);
    END IF;

    -- update outgoing opened feed
    --
    INSERT INTO social.feedback_aoi_feed_outgoing_opened
        SELECT * FROM social.feedback_aoi_feed_outgoing_closed WHERE feedback_task_id = NEW.id;
    DELETE FROM social.feedback_aoi_feed_outgoing_closed WHERE feedback_task_id = NEW.id;

    RETURN NULL;
END;
$$;


CREATE OR REPLACE FUNCTION social.feedback_task_outgoing_opened_experiment_update_trigger() RETURNS trigger
    LANGUAGE plpgsql
AS
$$
BEGIN
    IF (NEW.resolved_by IS NULL) THEN
        RETURN NEW;
    END IF;

    DELETE FROM social.feedback_task_outgoing_opened_experiment WHERE id = OLD.id;
    INSERT INTO social.feedback_task_outgoing_closed VALUES (NEW.*);
    RETURN NULL;
END;
$$;


CREATE OR REPLACE FUNCTION social.feedback_task_pending_update_trigger() RETURNS trigger
    LANGUAGE plpgsql
AS
$$
BEGIN
    IF (NEW.bucket != 'outgoing') THEN
        RETURN NEW;
    END IF;

    IF (NEW.resolved_by IS NULL) THEN

        IF (NEW.type IN ('address-experiment', 'entrance-experiment')) THEN
            INSERT INTO social.feedback_task_outgoing_opened_experiment VALUES (NEW.*);
        ELSE
            INSERT INTO social.feedback_task_outgoing_opened VALUES (NEW.*);
        END IF;

        -- update outgoing opened feed
        --
        INSERT INTO social.feedback_aoi_feed_outgoing_opened
            SELECT * FROM social.feedback_aoi_feed_pending WHERE feedback_task_id = NEW.id;

    ELSE

        INSERT INTO social.feedback_task_outgoing_closed VALUES (NEW.*);

        -- update outgoing closed feed
        --
        INSERT INTO social.feedback_aoi_feed_outgoing_closed
            SELECT * FROM social.feedback_aoi_feed_pending WHERE feedback_task_id = NEW.id;
    END IF;

    DELETE FROM social.feedback_task_pending WHERE id = NEW.id;
    DELETE FROM social.feedback_aoi_feed_pending WHERE feedback_task_id = NEW.id;

    RETURN NULL;
END;
$$;

-- 'closed_by' -> 'resolved_by'
--
DROP MATERIALIZED VIEW IF EXISTS social.feedback_task_distinct_source_type_mv;
DROP INDEX IF EXISTS social.feedback_task_distinct_source_type_mv_idx;

CREATE MATERIALIZED VIEW social.feedback_task_distinct_source_type_mv AS
SELECT
    source,
    type,
    hidden,
    workflow,
    (resolved_by IS NOT NULL) AS closed,
    min(created_at) AS min_created_at,
    max(created_at) AS max_created_at
FROM social.feedback_task
WHERE
    bucket = 'outgoing'
GROUP BY
    source,
    type,
    hidden,
    workflow,
    closed
WITH DATA;

-- In order to refresh this matview concurrently we must create unique index.
-- Refresh call is executed periodically in schedule_feedback_worker

CREATE UNIQUE INDEX feedback_task_distinct_source_type_mv_idx
ON social.feedback_task_distinct_source_type_mv
(source, type, hidden, workflow, closed, min_created_at, max_created_at);


-- 'fb.*' were causing problems with dropping columns
--
DROP MATERIALIZED VIEW IF EXISTS social.feedback_aoi_oldest_opened_task_mv;

CREATE MATERIALIZED VIEW social.feedback_aoi_oldest_opened_task_mv AS (
    WITH
        aoi_feedback_full AS (
            SELECT
                aoi.aoi_id,
                fb.id,
                fb.created_at,
                fb.type,
                fb.attrs,
                fb.source,
                fb.workflow
            FROM
                social.feedback_aoi_feed_outgoing_opened aoi
                    JOIN social.feedback_task_outgoing_opened fb
                         ON aoi.feedback_task_id = fb.id
        )
    SELECT *
    FROM
        (
            SELECT
                af_oldest.aoi_id,
                af_oldest.source,
                af_oldest.type,
                af_oldest.workflow,
                af_oldest.min_created_at,
                af.id,
                row_number()
                OVER (PARTITION BY af_oldest.aoi_id, af_oldest.source, af_oldest.type, af_oldest.workflow) AS row_id
            FROM
                (
                    SELECT
                        aoi_id,
                        source,
                        type,
                        workflow,
                        MIN(created_at) AS min_created_at
                    FROM
                        aoi_feedback_full
                    GROUP BY
                        aoi_id,
                        source,
                        type,
                        workflow
                ) AS af_oldest
                    JOIN aoi_feedback_full AS af ON
                        af_oldest.aoi_id = af.aoi_id AND
                        af_oldest.source = af.source AND
                        af_oldest.type = af.type AND
                        af_oldest.workflow = af.workflow AND
                        af_oldest.min_created_at = af.created_at
        ) sub
    WHERE row_id = 1
)
    WITH DATA;

-- In order to refresh this matview concurrently we must create unique index.
-- Refresh call is executed periodically in schedule_feedback_worker

CREATE UNIQUE INDEX feedback_aoi_oldest_opened_task_mv_idx
ON social.feedback_aoi_oldest_opened_task_mv
(aoi_id, source, type, workflow, min_created_at, id);


-- 'closed_at' -> 'resolved_at'. We don't recalculate all the hashes. They will
-- be updated lazily with modifications of rows and by feedback workers
-- (fbapi/st) over time.
--
CREATE OR REPLACE FUNCTION social.feedback_task_calc_hash_trigger() RETURNS trigger
    LANGUAGE plpgsql
AS
$$
BEGIN
    NEW.hash_value = MD5(ROW(NEW.hidden,NEW.resolved_at,NEW.deployed_at)::text);
    RETURN NEW;
END;
$$;

ALTER TABLE social.feedback_task_outgoing_opened
DROP CONSTRAINT check_null_closed_by,
DROP CONSTRAINT check_null_closed_at;

ALTER TABLE social.feedback_task_outgoing_closed
DROP CONSTRAINT check_not_null_closed_by,
DROP CONSTRAINT check_not_null_closed_at;
