CREATE TABLE IF NOT EXISTS acl.journal (
    txid bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now(),
    created_by bigint,
    table_name text NOT NULL,
    data_before jsonb,
    data_after jsonb
);

CREATE OR REPLACE FUNCTION acl.getCurrentUid() RETURNS bigint
LANGUAGE plpgsql
AS $$
BEGIN
   RETURN current_setting('wiki-acl.uid')::bigint;
EXCEPTION
   WHEN OTHERS THEN RETURN NULL;
END
$$;

CREATE OR REPLACE FUNCTION acl.setCurrentUid(IN uid bigint) RETURNS bigint
LANGUAGE plpgsql
AS $$
BEGIN
   PERFORM set_config('wiki-acl.uid', text(uid), true);
   RETURN current_setting('wiki-acl.uid')::bigint;
END
$$;

CREATE OR REPLACE FUNCTION acl.update_journal(
    IN table_name_arg text,
    IN data_before_arg json,
    IN data_after_arg json) RETURNS void
LANGUAGE plpgsql
AS $$
BEGIN
    INSERT INTO acl.journal (txid, created_by, table_name, data_before, data_after)
    VALUES (txid_current(), acl.getCurrentUid(), table_name_arg, data_before_arg, data_after_arg);
END
$$;

CREATE OR REPLACE FUNCTION acl.add_journaling(table_name text) RETURNS text
LANGUAGE plpgsql
AS $$
BEGIN
    EXECUTE format('
        CREATE OR REPLACE FUNCTION acl.journal_%1$s_update() RETURNS TRIGGER
        LANGUAGE plpgsql
        AS $body$
        BEGIN
          PERFORM acl.update_journal(''%1$s''::text, row_to_json(OLD.*), row_to_json(NEW.*));
          RETURN NEW;
        END
        $body$;

        CREATE OR REPLACE FUNCTION acl.journal_%1$s_delete() RETURNS TRIGGER
        LANGUAGE plpgsql
        AS $body$
        BEGIN
          PERFORM acl.update_journal(''%1$s''::text, row_to_json(OLD.*), ''{}''::json);
          RETURN NEW;
        END
        $body$;

        CREATE OR REPLACE FUNCTION acl.journal_%1$s_insert() RETURNS TRIGGER
        LANGUAGE plpgsql
        AS $body$
        BEGIN
          PERFORM acl.update_journal(''%1$s''::text, ''{}''::json, row_to_json(NEW.*));
          RETURN NEW;
        END
        $body$;

        DROP TRIGGER IF EXISTS acl_%1$s_update ON acl.%1$s;
        CREATE TRIGGER acl_%1$s_update
            AFTER UPDATE ON acl.%1$s
            FOR EACH ROW
            WHEN (OLD.* IS DISTINCT FROM NEW.*)
            EXECUTE PROCEDURE acl.journal_%1$s_update();

        DROP TRIGGER IF EXISTS acl_%1$s_delete ON acl.%1$s;
        CREATE TRIGGER acl_%1$s_delete
            AFTER DELETE ON acl.%1$s
            FOR EACH ROW
            EXECUTE PROCEDURE acl.journal_%1$s_delete();

        DROP TRIGGER IF EXISTS acl_%1$s_insert ON acl.%1$s;
        CREATE TRIGGER acl_%1$s_insert
            AFTER INSERT ON acl.%1$s
            FOR EACH ROW
            EXECUTE PROCEDURE acl.journal_%1$s_insert();',
        table_name);
    RETURN table_name;
END;
$$;

SELECT acl.add_journaling('ban_record');
SELECT acl.add_journaling('group');
SELECT acl.add_journaling('group_user');
SELECT acl.add_journaling('policy');
SELECT acl.add_journaling('restricted_users');
SELECT acl.add_journaling('role');
SELECT acl.add_journaling('role_permission');
SELECT acl.add_journaling('user');
