#include "access_checks.h"

#include "config.h"
#include "exception.h"
#include "magic_strings.h"
#include "moderation_status.h"
#include "user_info.h"

#include <maps/wikimap/mapspro/libs/acl/include/check_context.h>
#include <maps/wikimap/mapspro/libs/acl/include/exception.h>
#include <maps/wikimap/mapspro/libs/acl/include/subject_path.h>
#include <yandex/maps/wiki/common/json_helpers.h>

namespace maps::wiki::aclsrv {

namespace {

const acl::SubjectPath ACL_BASE_PATH("mpro/acl");
const acl::SubjectPath PERMISSIONS_ACL_PATH = ACL_BASE_PATH("permissions");
const acl::SubjectPath ASSIGN_ACL_PATH = ACL_BASE_PATH("assign");
const acl::SubjectPath BANS_ACL_PATH = ACL_BASE_PATH("ban");
const acl::SubjectPath GROUPS_ACL_PATH = ACL_BASE_PATH("groups");
const acl::SubjectPath ROLES_ACL_PATH = ACL_BASE_PATH("roles");
const acl::SubjectPath USERS_ACL_PATH = ACL_BASE_PATH("users");
const std::string IDM_SERVICE = "acl";
const std::string IDM_ADMIN_ROLE = "admin";

void checkAccess(const acl::SubjectPath& path, acl::UID uid, acl::Transaction& work)
{
    path.check(
        maps::wiki::acl::CheckContext(uid,
        std::vector<std::string>(), work, {acl::User::Status::Active}));
}
} // namespace

void checkAclPermissions(acl::UID uid, acl::Transaction& work, FromTest fromTest)
{
    acl::ACLGateway gw(work);
    const auto user = gw.user(uid);
    if (fromTest == FromTest::False && cfg()->enableIdmCheck()) {
        const auto staffLogin = user.staffLogin();
        if (staffLogin.empty()) {
            throw Error(Error::Status::Forbidden) << "User: " << user.login() << " has no staff account.";
        }
        if (!gw.userIDMRoles(staffLogin, IDM_SERVICE).count(IDM_ADMIN_ROLE)) {
            throw Error(Error::Status::Forbidden) << "Staff user: " << staffLogin << " has no IDM acl/admin role.";
        }
    }
    PERMISSIONS_ACL_PATH.check(
        maps::wiki::acl::CheckContext(user,
        std::vector<std::string>(), work, {acl::User::Status::Active}));
}

bool hasAccessToPermisions(acl::UID uid, acl::Transaction& work)
{
    return PERMISSIONS_ACL_PATH.isAllowed(
        maps::wiki::acl::CheckContext(uid,
        std::vector<std::string>(), work, {acl::User::Status::Active}));
}

void checkAclAssignPermissions(acl::UID uid, acl::Transaction& work)
{
   checkAccess(ASSIGN_ACL_PATH, uid, work);
}

bool hasAccessToAssign(acl::UID uid, acl::Transaction& work)
{
    return ASSIGN_ACL_PATH.isAllowed(
        maps::wiki::acl::CheckContext(uid,
        std::vector<std::string>(), work, {acl::User::Status::Active}));
}

void checkBanPermissions(acl::UID uid, acl::Transaction& work)
{
    checkAccess(BANS_ACL_PATH, uid, work);
}

void checkBanable(const acl::ACLGateway& aclgw, const acl::User& user)
{
    UserInfo userInfo(user, moderationStatus(aclgw, user));
    if (!userInfo.isBanable()) {
        throw acl::AccessDenied();
    }
}

void checkAccessToGroups(acl::UID uid, acl::Transaction& work)
{
    checkAccess(GROUPS_ACL_PATH, uid, work);
}

void checkAccessToRoles(acl::UID uid, acl::Transaction& work)
{
    checkAccess(ROLES_ACL_PATH, uid, work);
}

void checkAccessToUsers(acl::UID authorUid, acl::Transaction& work)
{
    if (authorUid) {// TODO: remove when uid (authorUid) added to all /users handlers NMAPS-13898
        checkAccess(USERS_ACL_PATH, authorUid, work);
    }
}

CanAssign
getUserCanAssign(acl::UID uid, acl::Transaction& work)
{
    CanAssign result {.uid = uid};
    try {
        if (!hasAccessToAssign(uid, work)) {
            return result;
        }
        if (hasAccessToPermisions(uid, work)) {
            result.any = CanAssign::Any::True;
        } else {
            const auto user = acl::ACLGateway(work).user(uid);
            for (const auto& pol : user.allPolicies()) {
                for (const auto& role : pol.role().canAssignRoles()) {
                    if (!result.roleIds.count(role.id())) {
                        result.roleIds.insert(role.id());
                    }
                }
                for (const auto& group : pol.role().canAssignGroups()) {
                    if (!result.groupIds.count(group.id())) {
                        result.groupIds.insert(group.id());
                    }
                }
            }
        }
    } catch (const acl::AccessDenied&) {
        return CanAssign {.uid = uid};
    }
    return result;
}

} // namespace maps::wiki::aclsrv
