#include <maps/wikimap/mapspro/services/autocart/libs/detection/include/graph_pgn_extract.h>

#include <maps/wikimap/mapspro/services/autocart/libs/detection/include/utils.h>
#include <maps/wikimap/mapspro/services/autocart/libs/detection/include/graph.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <contrib/libs/opencv/include/opencv2/imgproc.hpp>

#include <iostream>
#include <set>
#include <map>
#include <vector>
#include <iterator>
#include <algorithm>


namespace maps {
namespace wiki {
namespace autocart {

namespace {

/**
    Extract local maximum from img.
*/
void extractLocalMaximum(const cv::Mat &img, cv::Mat &limg, double vertsMinThreshold)
{
    cv::Mat imgth;
    if (FLT_EPSILON < vertsMinThreshold)
        cv::threshold(img, imgth, vertsMinThreshold, 0., cv::THRESH_TOZERO);
    else
        imgth = img;

    cv::Mat mask = cv::Mat::ones(imgth.size(), CV_8UC1) * 255;
    limg = cv::Mat::zeros(imgth.size(), CV_8UC1);

    double maxVal; cv::Point maxLoc;
    for (;;)
    {
        cv::minMaxLoc(imgth, NULL, &maxVal, NULL, &maxLoc, mask);
        if (maxVal == 0)
            break;

        cv::Point2d ptSum(0.0, 0.0);
        double valSum = 0.;
        std::vector<cv::Point> pts; pts.push_back(maxLoc);
        mask.at<uchar>(maxLoc) = 0;
        ptSum.x = (double)maxLoc.x * (double)maxVal;
        ptSum.y = (double)maxLoc.y * (double)maxVal;
        valSum += (double)maxVal;
        while (!pts.empty())
        {
            cv::Point pt = pts.back(); pts.pop_back();
            //left
            {
                float val = imgth.at<float>(pt);
                for (int i = pt.x - 1; i >= 0; i--)
                {
                    if (0 == mask.at<uchar>(pt.y, i))
                        break;
                    if (val < imgth.at<float>(pt.y, i))
                        break;
                    val = imgth.at<float>(pt.y, i);
                    mask.at<uchar>(pt.y, i) = 0;
                    pts.emplace_back(i, pt.y);

                    const double w = (double)val;
                    ptSum.x += (double)   i * w;
                    ptSum.y += (double)pt.y * w;
                    valSum += w;
                }
            }
            //right
            {
                float val = imgth.at<float>(pt);
                for (int i = pt.x + 1; i < imgth.cols; i++)
                {
                    if (0 == mask.at<uchar>(pt.y, i))
                        break;
                    if (val < imgth.at<float>(pt.y, i))
                        break;
                    val = imgth.at<float>(pt.y, i);
                    mask.at<uchar>(pt.y, i) = 0;
                    pts.emplace_back(i, pt.y);

                    const double w = (double)val;
                    ptSum.x += (double)   i * w;
                    ptSum.y += (double)pt.y * w;
                    valSum += w;
                }
            }
            //up
            {
                float val = imgth.at<float>(pt);
                for (int i = pt.y - 1; i > 0; i--)
                {
                    if (0 == mask.at<uchar>(i, pt.x))
                        break;
                    if (val < imgth.at<float>(i, pt.x))
                        break;
                    val = imgth.at<float>(i, pt.x);
                    mask.at<uchar>(i, pt.x) = 0;
                    pts.emplace_back(pt.x, i);

                    const double w = (double)val;
                    ptSum.x += (double)pt.x * w;
                    ptSum.y += (double)   i * w;
                    valSum += w;
                }
            }
            //down
            {
                float val = imgth.at<float>(pt);
                for (int i = pt.y + 1; i < imgth.rows; i++)
                {
                    if (0 == mask.at<uchar>(i, pt.x))
                        break;
                    if (val < imgth.at<float>(i, pt.x))
                        break;
                    val = imgth.at<float>(i, pt.x);
                    mask.at<uchar>(i, pt.x) = 0;
                    pts.emplace_back(pt.x, i);

                    const double w = (double)val;
                    ptSum.x += (double)pt.x * w;
                    ptSum.y += (double)   i * w;
                    valSum += w;
                }
            }
        }
        if (valSum > DBL_EPSILON)
            limg.at<uchar>(ptSum / valSum) = 255;
    }
}

void extractCoords(const cv::Mat &limg, std::vector<cv::Point> &coords)
{
    const int rows = limg.rows;
    const int cols = limg.cols;
    for (int row = 0; row < rows; row++)
    {
        const uchar *ptr = limg.ptr<uchar>(row);
        for (int col = 0; col < cols; col++)
        {
            if (ptr[col])
                coords.emplace_back(col, row);
        }
    }
}

void extractVerticesCoords(const cv::Mat &vertData, std::vector<cv::Point> &vertices, double vertsMinThreshold)
{
    cv::Mat limg;
    extractLocalMaximum(vertData, limg, vertsMinThreshold);
    extractCoords(limg, vertices);
}

void nonMaximumSuppresion(const cv::Mat &vertData, std::vector<cv::Point> &vertices, int cellSz)
{
    const int cellSzSq = cellSz * cellSz;
    for (int i = 0; i < (int)vertices.size() - 1; i++)
    {
        const cv::Point &pt1 = vertices[i];
        const float val1 = vertData.at<float>(pt1);
        bool del = false;
        for (int j = i + 1; j < (int)vertices.size(); j++)
        {
            const cv::Point &pt2 = vertices[j];
            const float val2 = vertData.at<float>(pt2);
            const cv::Point diff = pt2 - pt1;
            if (cellSzSq < diff.dot(diff))
                continue;

            if (val1 < val2)
            {
                del = true;
                break;
            }
            vertices.erase(vertices.begin() + j);
            j--;
        }
        if (del)
        {
            vertices.erase(vertices.begin() + i);
            i--;
        }
    }
}

double lineWeight(const cv::Mat &edgeData, const cv::Point &pt1, const cv::Point &pt2)
{
    REQUIRE(edgeData.type() == CV_32FC1, "Matrix with ropabilities of edges should be 32FC1 type");
    cv::LineIterator it(edgeData, pt1, pt2, 8);
    double edgesWeight = 0.;
    for (int k = 0; k < it.count; k++, ++it)
    {
        edgesWeight += (double)(*(float *)*it);
    }
    return (edgesWeight / it.count);
}

double deltaRightAngle(double angle)
{
    return abs(angle - floor(2. * (angle + CV_PI / 4.) / CV_PI) * CV_PI / 2.);
}

class CoordGraph
    : public Graph<cv::Point, double>
{
private:
    struct WeightedEdgesCycle
    {
        EdgesPath cycle;
        double weight;
    };

    struct WeightedEdgesCycleComparer {
        bool operator() (const WeightedEdgesCycle& lhs, const WeightedEdgesCycle& rhs) const
        {
            return lhs.weight > rhs.weight;
        }
    };

    using WeightedEdgesCyclesSet = std::set < WeightedEdgesCycle, WeightedEdgesCycleComparer >;
public:
    CoordGraph() {}

    ~CoordGraph() {}

    /*
        vertices - propability of the pixel into vertex of building
        edges    - propability of the pixel into edge of building
    */
    bool importData(const cv::Mat &vertData, const cv::Mat &edgeData, const ExtractPolygonsParams &params)
    {
        REQUIRE(vertData.size() == edgeData.size(),
                "Vertex propability data and edge propability data have to have same size");
        clear();

        importVertices(vertData, params.vertsParams);
        importEdges(edgeData, params.edgesParams);

        return true;
    }
    void updateEdges(const cv::Mat &edgeData, const ExtractPolygonsParams &params)
    {
        clearEdges();
        importEdges(edgeData, params.edgesParams);
    }

    /**
        remove vertices
        maxAngle - in degree
    */
    void removeKeenVertices(double maxAngle)
    {
        bool doit = true;
        while (doit)
        {
            doit = false;
            for (auto it = beginVertices(); it != endVertices(); it++)
            {
                if (it->second.edges.size() <= 1)
                {
                    eraseVertex(it->first);
                    doit = true;
                    break;
                }
                if (it->second.edges.size() == 2)
                {
                    auto ite = it->second.edges.begin();
                    const int e1 = (*ite);
                    ite++;
                    const int e2 = (*ite);
                    auto ite1 = findEdge(e1)->second;
                    auto ite2 = findEdge(e2)->second;

                    const int v1 = ite1.getOverVertex(it->first);
                    const int v2 = ite2.getOverVertex(it->first);
                    const double angle = calcAngle(findVertex(v1)->second.data, findVertex(it->first)->second.data, findVertex(v2)->second.data);
                    if (angle < maxAngle)
                    {
                        eraseVertex(it->first);
                        doit = true;
                        break;
                    }
                }
            }
        }
    }

    void removeThinPolygon(const ThinPolygonsParams &params)
    {
        //remove thin and long triangles (one vertex very near edges) by erasing longest edge
        const double maxDistSq = params.maxVertEdgeDist * params.maxVertEdgeDist;
        for (auto ite = beginEdges(); ite != endEdges();)
        {
            const auto itv1 = findVertex(ite->second.vertex1);
            const auto itv2 = findVertex(ite->second.vertex2);

            const cv::Point &coordv1 = itv1->second.data;
            const cv::Point &coordv2 = itv2->second.data;

            bool del = false;
            for (auto itv = beginVertices(); itv != endVertices(); itv++)
            {
                if (itv->first == itv1->first || itv->first == itv2->first)
                    continue;
                const cv::Point &coordv = itv->second.data;

                const cv::Point vv1 = coordv - coordv1;
                const cv::Point v2v1 = coordv2 - coordv1;
                const double t = vv1.ddot(v2v1) / v2v1.ddot(v2v1);
                if (t < params.leftRange || t > params.rightRange)
                    continue;

                const cv::Point2d diff = coordv1 + t * v2v1 - coordv;
                if (diff.ddot(diff) < maxDistSq)
                {
                    // пробовал при отсутствии общего ребра его добавлять получилось плохо
                    // потому что вариантов два
                    //   1. добавляем только если ребро имеет вес выше порогового,
                    //      но оно тогда и так бы уже было добавлено.
                    //   2. добавляем в любом случае, даже при недостаточном весе -
                    //      мы т.о. оказываем воздействие на взвешивание циклов,
                    //      потому что получаем ребра, которые слабые с т.з. веса
                    //      но могут образовать "хороший" угол и не понятно, почему
                    //      мы делаем это для ребер из тонких треугольников?
                    // поэтому пока решение удаляем ребро, только если не нарушается связность
                    //del = hasCommonEdge(itv1->second, itv->second) && hasCommonEdge(itv2->second, itv->second);
                    // может случиться, что itv1 и itv не связаны ребром, но связаны через промежуточную вершину
                    // поэтому вместо hasCommonEdge пользуем isVertsConnected
                    del = isVertsConnected(itv1->first, itv->first, ite->first) &&
                          isVertsConnected(itv2->first, itv->first, ite->first);
                    break;
                }
            }
            if (del)
                ite = eraseEdge(ite);
            else
                ite++;
        }
    }

    void removeIntersectedEdges()
    {// filter by intersection
        auto ite1 = beginEdges();
        while (ite1 != endEdges())
        {
            const cv::Point &e1v1 = findVertex(ite1->second.vertex1)->second.data;
            const cv::Point &e1v2 = findVertex(ite1->second.vertex2)->second.data;

            bool del1 = false;
            auto ite2 = ite1;
            ite2++;
            while (ite2 != endEdges())
            {
                bool del2 = false;
                if (ite1->second.vertex1 != ite2->second.vertex1 &&
                    ite1->second.vertex1 != ite2->second.vertex2 &&
                    ite1->second.vertex2 != ite2->second.vertex1 &&
                    ite1->second.vertex2 != ite2->second.vertex2)
                {
                    const cv::Point &e2v1 = findVertex(ite2->second.vertex1)->second.data;
                    const cv::Point &e2v2 = findVertex(ite2->second.vertex2)->second.data;
                    if (isSegmentIntersect(e1v1, e1v2, e2v1, e2v2))
                    {
                        if (ite1->second.data < ite2->second.data)
                        {
                            del1 = true;
                            break;
                        }
                        del2 = true;
                    }
                }
                if (del2)
                    ite2 = eraseEdge(ite2);
                else
                    ++ite2;
            }
            if (del1)
                ite1 = eraseEdge(ite1);
            else
                ++ite1;
        }
    }

    void extractPolygons(const cv::Mat &segdata, std::vector< std::vector<cv::Point> > &polygons, const ExtractPolygonsParams &params)
    {
        std::vector<WeightedEdgesCyclesSet> cyclesSet;
        extractCycles(segdata, cyclesSet, params);

        std::vector<EdgesPath> best;
        for (const auto &cycle : cyclesSet)
        {
            getBestEdgesCycles(cycle, segdata.size(), best, params.bestCycleParams);
        }

        polygons.resize(best.size());
        for (size_t i = 0; i < best.size(); i++)
        {
            convertEdgeCycleToCoords(best[i], polygons[i]);
        }
    }
private:
    void importVertices(const cv::Mat &vertData, const ExtractVerticesParams &params)
    {
        std::vector<cv::Point> vertices;
        extractVerticesCoords(vertData, vertices, params.minThreshold);
        nonMaximumSuppresion(vertData, vertices, params.nmsCellSz);
        for (size_t i = 0; i < vertices.size(); i++)
        {
            addVertex(vertices[i]);
        }
    }

    void importEdges(const cv::Mat &edgeData, const ExtractEdgesParams &params)
    {
        for (auto itv1 = beginVertices(); itv1 != endVertices(); ++itv1)
        {
            const cv::Point &pt1 = itv1->second.data;
            auto itv2 = itv1; ++itv2;
            for (; itv2 != endVertices(); ++itv2)
            {
                const cv::Point &pt2 = itv2->second.data;
                const double length = cv::norm(pt2 - pt1);
                if (length > params.maxLength ||
                    length < params.minLength)
                    continue;

                const double weight = lineWeight(edgeData, pt1, pt2);
                if (weight > params.minWeightThreshold)
                    addEdge(itv1->first, itv2->first, weight);
            }
        }
    }
private:
    double edgesAngle(const Edge &edge1, const Edge &edge2) const
    {
        int centerId = INVALID_INDEX;
        int vertexId1 = INVALID_INDEX;
        int vertexId2 = INVALID_INDEX;
        if (edge1.vertex1 == edge2.vertex1)
        {
            centerId = edge1.vertex1;
            vertexId1 = edge1.vertex2;
            vertexId2 = edge2.vertex2;
        }
        else if (edge1.vertex1 == edge2.vertex2)
        {
            centerId = edge1.vertex1;
            vertexId1 = edge1.vertex2;
            vertexId2 = edge2.vertex1;
        }
        else if (edge1.vertex2 == edge2.vertex1)
        {
            centerId = edge1.vertex2;
            vertexId1 = edge1.vertex1;
            vertexId2 = edge2.vertex2;
        }
        else if (edge1.vertex2 == edge2.vertex2)
        {
            centerId = edge1.vertex2;
            vertexId1 = edge1.vertex1;
            vertexId2 = edge2.vertex1;
        }
        else
            REQUIRE(false, "edges don't connected");

        return calcAngle(findVertex(vertexId1)->second.data, findVertex(centerId)->second.data, findVertex(vertexId2)->second.data);
    }

    void convertEdgeCycleToCoords(const EdgesPath &cycle, std::vector<cv::Point> &coords) const
    {
        coords.resize(cycle.size());
        auto zeroEdge = findEdge(cycle[0]);
        int lastVert = findEdge(cycle[1])->second.getOverVertex(zeroEdge->second.vertex1);
        if (lastVert != INVALID_INDEX)
        {
            coords[0] = findVertex(zeroEdge->second.vertex2)->second.data;
            lastVert = zeroEdge->second.vertex1;
        }
        else
        {
            coords[0] = findVertex(zeroEdge->second.vertex1)->second.data;
            lastVert = zeroEdge->second.vertex2;
        }
        for (size_t j = 1; j < cycle.size(); ++j)
        {
            coords[j] = findVertex(lastVert)->second.data;
            lastVert = findEdge(cycle[j])->second.getOverVertex(lastVert);
            REQUIRE(INVALID_INDEX != lastVert, "Invalid vertices");
        }
    }

    void sumEdgesCycle(const EdgesPath &cycle1, const EdgesPath &cycle2, EdgesPath &sumCycle) const
    {
        std::set<int> result;
        for (int i : cycle1)
            result.insert(i);
        for (int i : cycle2)
        {
            auto it = result.find(i);
            if (it == result.end())
                result.insert(i);
            else
                result.erase(it);
        }
        if (result.size() == cycle1.size() + cycle2.size())
            return; //no common elements

        auto it = result.begin();
        int lastEdgeId = *it;
        int lastVertexId = INVALID_INDEX;
        {
            sumCycle.push_back(lastEdgeId);
            result.erase(it);
            const auto lastEdge = findEdge(lastEdgeId);
            lastVertexId = lastEdge->second.vertex1;
        }
        while (!result.empty())
        {
            const auto vert = findVertex(lastVertexId);
            bool found = false;
            for (int edge : vert->second.edges)
            {
                it = result.find(edge);
                if (it != result.end())
                {
                    lastEdgeId = *it;
                    sumCycle.push_back(lastEdgeId);
                    result.erase(it);
                    const auto lastEdge = findEdge(lastEdgeId);
                    lastVertexId = lastEdge->second.getOverVertex(lastVertexId);
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                sumCycle.clear();
                return;
            }
        }
    }

    void rasterizeEdgeCycle(const EdgesPath &cycle, cv::Rect &roi, cv::Mat &cycleMask) const
    {
        std::vector <std::vector<cv::Point>> coords(1);
        convertEdgeCycleToCoords(cycle, coords[0]);
        int minx = INT_MAX;
        int maxx = INT_MIN;
        int miny = INT_MAX;
        int maxy = INT_MIN;
        for (const auto &pt : coords[0])
        {
            minx = MIN(pt.x, minx);
            maxx = MAX(pt.x, maxx);
            miny = MIN(pt.y, miny);
            maxy = MAX(pt.y, maxy);
        }
        roi = cv::Rect(minx, miny, maxx - minx + 1, maxy - miny + 1);
        cycleMask = cv::Mat::zeros(roi.size(), CV_8UC1);
        for (size_t i = 0; i < coords[0].size(); i++)
        {
            coords[0][i].x -= minx;
            coords[0][i].y -= miny;
        }
        cv::fillPoly(cycleMask, coords, cv::Scalar(1., 1., 1.));
    }

    // is there vertices with more than two edges
    bool hasVertDegreeGreatTwo(const EdgesPath &cycle) const
    {
        std::map<int, int> vertsDegree;
        for (int i : cycle)
        {
            const auto edge = findEdge(i);
            auto it1 = vertsDegree.find(edge->second.vertex1);
            if (it1 != vertsDegree.end())
            {
                it1->second++;
                if (it1->second > 2)
                    return true;
            }
            else
                vertsDegree[edge->second.vertex1] = 1;
            auto it2 = vertsDegree.find(edge->second.vertex2);
            if (it2 != vertsDegree.end())
            {
                it2->second++;
                if (it2->second > 2)
                    return true;
            }
            else
                vertsDegree[edge->second.vertex2] = 1;
        }
        return false;
    }

    double weightCycle(const EdgesPath &cycle, const WeightCycleParams &params) const
    {
        if (cycle.size() < 4)
            return 0.;
        double sudelta = 0.;
        auto zeroEdge = findEdge(cycle[0]);
        auto lastEdge = zeroEdge;
        double weight = lastEdge->second.data;
        for (size_t i = 1; i < cycle.size(); i++)
        {
            auto curEdge = findEdge(cycle[i]);
            const double angle = edgesAngle(lastEdge->second, curEdge->second);
            if (angle < params.smallAngleMax)
                sudelta += 1.0 / (params.smallAnglePenaltyEps + angle);
            else
                sudelta += deltaRightAngle(angle);
            weight += curEdge->second.data;
            lastEdge = curEdge;
        }
        const double angle = edgesAngle(lastEdge->second, zeroEdge->second);
        if (angle < params.smallAngleMax)
            sudelta += 1.0 / (params.smallAnglePenaltyEps + angle);
        sudelta += deltaRightAngle(angle);

        weight /= (cycle.size() + params.angleDeltaCoefPenalty * sudelta);
        weight -= (cycle.size() > 4) ? params.nonQuadPenalty * (cycle.size() - 4) : 0.0;
        return weight;
    }

    double weightCycle(const EdgesPath &cycle, const cv::Mat &segm, const WeightCycleParams &params) const
    {
        if (cycle.size() < 4)
            return 0.;
        //TODO OPT
        const double weight = weightCycle(cycle, params);

        cv::Rect roi;
        cv::Mat cycleMask;
        rasterizeEdgeCycle(cycle, roi, cycleMask);
        const int cycleArea = cv::countNonZero(cycleMask);
        if (cycleArea < params.minArea)
            return 0.0;

        if (segm.empty())
            return weight;
        cycleMask = cycleMask.mul(segm(roi));
        const int validArea = cv::countNonZero(cycleMask);
        return weight + params.segmAreaCoeff * ((double)validArea / (double)cycleArea);
    }

    void extractCycles(const cv::Mat &segm,
                       const EdgesPath &baseCycle,
                       const std::set<size_t> &used,
                       const std::vector<EdgesPath> &cycles,
                       const WeightCycleParams &params,
                       int restLvl,
                       WeightedEdgesCyclesSet &cyclesSet) const
    {
        restLvl--;
        for (size_t i = 0; i < cycles.size(); i++)
        {
            if (used.find(i) != used.end())
                continue;
            const EdgesPath &cycle = cycles[i];
            WeightedEdgesCycle sucycle;
            sumEdgesCycle(baseCycle, cycle, sucycle.cycle);
            if (sucycle.cycle.empty())
                continue;
            if (!hasVertDegreeGreatTwo(sucycle.cycle))
            {
                sucycle.weight = weightCycle(sucycle.cycle, segm, params);
                cyclesSet.insert(sucycle);
            }
            if (0 < restLvl)
            {
                std::set<size_t> usedNew = used;
                usedNew.insert(i);
                extractCycles(segm, sucycle.cycle, usedNew, cycles, params, restLvl, cyclesSet);
            }
        }
    }

    void extractCycles(const cv::Mat &segm,
                       std::vector<WeightedEdgesCyclesSet> &arCyclesSet,
                       const ExtractPolygonsParams &params) const
    {
        REQUIRE(params.extractLevelsCount > 2, "extractLevelsCount should be great than 2");
        std::vector<std::vector<EdgesPath>> arcycles = extractFundamentalCyclesEdges();
        arCyclesSet.resize(arcycles.size());

        for (size_t c = 0; c < arcycles.size(); c++)
        {
            const std::vector<EdgesPath> &cycles = arcycles[c];
            if (0 == cycles.size())
                continue;
            WeightedEdgesCyclesSet &cyclesSet = arCyclesSet[c];

            for (size_t i = 0; i < cycles.size(); i++)
            {
                WeightedEdgesCycle cycle;
                cycle.cycle = cycles[i];
                cycle.weight = weightCycle(cycle.cycle, segm, params.weightCycleParams);
                cyclesSet.emplace(cycle);
            }
            for (size_t i0 = 0; i0 < cycles.size() - 1; i0++)
            {
                const EdgesPath &cycle1 = cycles[i0];
                for (size_t i1 = i0 + 1; i1 < cycles.size(); i1++)
                {
                    const EdgesPath &cycle2 = cycles[i1];
                    WeightedEdgesCycle sucycle;
                    sumEdgesCycle(cycle1, cycle2, sucycle.cycle);
                    if (sucycle.cycle.empty())
                        continue;
                    if (!hasVertDegreeGreatTwo(sucycle.cycle))
                    {
                        sucycle.weight = weightCycle(sucycle.cycle, segm, params.weightCycleParams);
                        cyclesSet.insert(sucycle);
                    }

                    std::set<size_t> used({i0, i1});
                    extractCycles(segm, sucycle.cycle, used, cycles, params.weightCycleParams, params.extractLevelsCount - 2, cyclesSet);
                }
            }
        }
    }

    void getBestEdgesCycles(const WeightedEdgesCyclesSet &cyclesSet,
                            const cv::Size &imgsz, std::vector<EdgesPath> &best,
                            const BestCycleParams &params) const
    {
        size_t graphEdgesCnt = edgesSize();
        cv::Mat img = cv::Mat::zeros(imgsz, CV_8UC1);
        cv::Rect roi;
        cv::Mat cycleMask(imgsz, CV_8UC1); // allocate buffer
        cv::Mat temp(imgsz, CV_8UC1); // allocate buffer
        std::set<int> edges;
        for (const auto &cycle : cyclesSet)
        {
            if (cycle.weight < params.minCycleWeight)
                break;

            rasterizeEdgeCycle(cycle.cycle, roi, cycleMask);

            const int cycleArea = cv::countNonZero(cycleMask);
            temp = cycleMask.mul(img(roi));
            const int intersectArea = cv::countNonZero(temp);
            if ((double)intersectArea > params.minRelativeIntersectionArea * (double)cycleArea ||
                (double)intersectArea > params.minIntersectionArea)
                continue;

            int newEdges = 0;
            for (const auto &edge : cycle.cycle)
            {
                if (edges.find(edge) == edges.end())
                    newEdges++;
            }
            if ((double)newEdges < (double)cycle.cycle.size() * params.newEdgesPercent)
                continue;

            img(roi).setTo(1, cycleMask);

            for (const auto &edge : cycle.cycle)
            {
                edges.insert(edge);
            }
            best.push_back(cycle.cycle);
            if (edges.size() == graphEdgesCnt)
                break;
        }
    }
};
}

std::vector<std::vector<cv::Point> >
extractPolygons(const cv::Mat &vertData,
                const cv::Mat &edgeData,
                const cv::Mat &segdata,
                const ExtractPolygonsParams &params)
{
    constexpr double edgesMinWeightThresholdCoef = 1.5;
    constexpr int edgesMinWeightThresholdIncLvls = 10;
    double edgesMinWeightThresholdInc = 1.0;

    std::vector<std::vector<cv::Point> > polygons;
    ExtractPolygonsParams localParams = params;
    if (params.edgesParams.minWeightThreshold < 0.) {
        localParams.edgesParams.minWeightThreshold = edgesMinWeightThresholdCoef * cv::mean(edgeData)[0];
        if (localParams.edgesParams.minWeightThreshold > 1.0 - FLT_EPSILON) {
            WARN() << "Mean of edges data is too big, extractPolygons stoped";
            return polygons;
        } else if (localParams.edgesParams.minWeightThreshold < FLT_EPSILON) {
            WARN() << "Mean of edges data is zero, extractPolygons stoped";
            return polygons;
        }
        edgesMinWeightThresholdInc =
            pow(1. / localParams.edgesParams.minWeightThreshold,
                1.0 / (double)edgesMinWeightThresholdIncLvls);
        INFO() << "Initialize edges weights to "
               << localParams.edgesParams.minWeightThreshold;
    }

    CoordGraph graph;
    if (!graph.importData(vertData, edgeData, localParams))
        return polygons;

    for (int i = 0; i < edgesMinWeightThresholdIncLvls; i++)
    {
        if (0. < localParams.keenVerticesMaxAngle)
            graph.removeKeenVertices(localParams.keenVerticesMaxAngle);
        else
            graph.eraseLeaves();
        if (0. < localParams.thinPolygonsParams.maxVertEdgeDist)
            graph.removeThinPolygon(localParams.thinPolygonsParams);
        if (localParams.removeIntersectEdges)
            graph.removeIntersectedEdges();
        if (params.edgesParams.minWeightThreshold < 0.)
        {
            std::vector<CoordGraph::GraphIndices> forest = graph.extractSpanningForest();
            int maxEdges = 0;
            for (size_t i = 0; i < forest.size(); i++)
            {
                maxEdges = std::max((int)forest[i].edgesSize(), maxEdges);
            }
            INFO() << "There are " << maxEdges
                   << " in one of the connected components.";
            if (maxEdges > localParams.edgesParams.maxEdgesInConComponent)
            {
                localParams.edgesParams.minWeightThreshold *= edgesMinWeightThresholdInc;
                INFO() << "Change edges weights to "
                       << localParams.edgesParams.minWeightThreshold;
                graph.updateEdges(edgeData, localParams);
                continue;
            }
        }
        graph.extractPolygons(segdata, polygons, localParams);
        break;
    }
    return polygons;
}
} //namespace autocart
} //namespace wiki
} //namespace maps
