#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/polygon_processing.h>

#include <maps/libs/log8/include/log8.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/distance.h>

#include <contrib/libs/geos/include/geos/geom/Geometry.h>
#include <contrib/libs/geos/include/geos/geom/Polygon.h>
#include <contrib/libs/geos/include/geos/geom/MultiPolygon.h>
#include <contrib/libs/geos/include/geos/geom/MultiLineString.h>
#include <contrib/libs/geos/include/geos/operation/union/CascadedUnion.h>

#include <vector>
#include <utility>
#include <algorithm>
#include <memory>
#include <optional>
#include <iostream>

namespace maps {
namespace wiki {
namespace autocart {

std::vector<geolib3::Polygon2>
extractPolygons(const geos::geom::Geometry& geom) {
    std::vector<geolib3::Polygon2> result;
    if (geom.getGeometryTypeId() == geos::geom::GEOS_POLYGON) {
        const geos::geom::Polygon* ptr = dynamic_cast<const geos::geom::Polygon*>(&geom);
        if (ptr->getArea() < geolib3::EPS * geolib3::EPS) {
            WARN() << "Skip polygon with zero area in exctractPolygons()";
        } else {
            result.push_back(geolib3::internal::geos2geolibGeometry(ptr));
        }
    } else if (geom.getGeometryTypeId() == geos::geom::GEOS_MULTIPOLYGON) {
        geolib3::MultiPolygon2 multipolygon
            = geolib3::internal::geos2geolibGeometry(
                dynamic_cast<const geos::geom::MultiPolygon*>(&geom)
            );
        for (size_t i = 0; i < multipolygon.polygonsNumber() ; i++) {
            result.push_back(multipolygon.polygonAt(i));
        }
    } else if (geom.getNumGeometries() > 1) {
        for (size_t i = 0; i < geom.getNumGeometries(); ++i) {
            const geos::geom::Geometry* thisGeom = geom.getGeometryN(i);
            std::vector<geolib3::Polygon2> extracted = extractPolygons(*thisGeom);
            std::move(
                extracted.begin(), extracted.end(), std::back_inserter(result)
            );
        }
    }
    return result;
}

geolib3::Polygon2 rotatePolygon(const geolib3::Polygon2& polygon,
                                float angleDegree,
                                const geolib3::Point2& center) {
    std::vector<geolib3::Point2> rotatedPts;

    const double cosa = cos(angleDegree * M_PI / 180.f);
    const double sina = sin(angleDegree * M_PI / 180.f);

    for (size_t i = 0; i < polygon.pointsNumber(); i++) {
        const geolib3::Point2 src = polygon.pointAt(i);
        double rotatedX, rotatedY;
        // move center to origin
        geolib3::Point2 temp(src.x() - center.x(), src.y() - center.y());

        // rotate
        rotatedX = temp.x() * cosa - temp.y() * sina;
        rotatedY = temp.x() * sina + temp.y() * cosa;

        // move center to old position
        rotatedX += center.x();
        rotatedY += center.y();
        rotatedPts.emplace_back(rotatedX, rotatedY);
    }

    return geolib3::Polygon2(rotatedPts);
}

geolib3::Polygon2 rotatePolygon(const geolib3::Polygon2& polygon,
                                float angleDegree) {
    geolib3::Point2 centroid = polygon.exteriorRing().findCentroid();
    return rotatePolygon(polygon, angleDegree, centroid);
}

geolib3::MultiPolygon2
intersectPolygons(
    const geolib3::Polygon2& polygon1,
    const geolib3::Polygon2& polygon2)
{
    auto geosPolygon1 = geolib3::internal::geolib2geosGeometry(polygon1);
    auto geosPolygon2 = geolib3::internal::geolib2geosGeometry(polygon2);
    std::unique_ptr<geos::geom::Geometry> intersectedGeom(
        geosPolygon1->intersection(geosPolygon2.get())
    );
    return geolib3::MultiPolygon2(extractPolygons(*intersectedGeom));
}

geolib3::MultiPolygon2
intersectMultiPolygons(
    const geolib3::MultiPolygon2& multiPolygon1,
    const geolib3::MultiPolygon2& multiPolygon2)
{
    auto geosMultiPolygon1 = geolib3::internal::geolib2geosGeometry(multiPolygon1);
    auto geosMultiPolygon2 = geolib3::internal::geolib2geosGeometry(multiPolygon2);
    std::unique_ptr<geos::geom::Geometry> intersectedGeom(
        geosMultiPolygon1->intersection(geosMultiPolygon2.get())
    );

    return geolib3::MultiPolygon2(extractPolygons(*intersectedGeom));
}

geolib3::MultiPolygon2
mergePolygons(const std::vector<geolib3::Polygon2>& polygons) {
    if (polygons.empty()) {
        return {};
    }
    std::shared_ptr<const geos::geom::Geometry> mergedGeom
        = geolib3::internal::geolib2geosGeometry(polygons.front());
    for (size_t i = 1; i < polygons.size(); i++) {
        std::shared_ptr<const geos::geom::Polygon> geom
            = geolib3::internal::geolib2geosGeometry(polygons[i]);
        mergedGeom = mergedGeom->Union(geom.get());
    }
    return geolib3::MultiPolygon2(extractPolygons(*mergedGeom));
}

geolib3::MultiPolygon2
cascadedMergePolygons(const std::vector<geolib3::Polygon2>& polygons) {
    std::vector<geos::geom::Geometry*> ptrs;
    for (const geolib3::Polygon2& polygon : polygons) {
        // return shared_ptr that is contained in geolib3::Polygon2
        std::shared_ptr<const geos::geom::Polygon> geosPtr
            = geolib3::internal::geolib2geosGeometry(polygon);
        ptrs.push_back(
            const_cast<geos::geom::Geometry*>(
                static_cast<const geos::geom::Geometry*>(geosPtr.get())
            )
        );
    }
    std::unique_ptr<geos::geom::Geometry> united(
        geos::operation::geounion::CascadedUnion::Union(&ptrs)
    );
    return geolib3::MultiPolygon2(extractPolygons(*united));
}

geolib3::MultiPolygon2
mergeMultiPolygons(const std::vector<geolib3::MultiPolygon2>& multiPolygons) {
    if (multiPolygons.empty()) {
        return {};
    }
    std::shared_ptr<const geos::geom::Geometry> mergedGeom
        = geolib3::internal::geolib2geosGeometry(multiPolygons.front());
    for (size_t i = 1; i < multiPolygons.size(); i++) {
        std::shared_ptr<const geos::geom::MultiPolygon> geom
            = geolib3::internal::geolib2geosGeometry(multiPolygons[i]);
        mergedGeom = mergedGeom->Union(geom.get());
    }
    return geolib3::MultiPolygon2(extractPolygons(*mergedGeom));
}

geolib3::MultiPolygon2
cascadedMergeMultiPolygons(const std::vector<geolib3::MultiPolygon2>& multiPolygons) {
    std::vector<geos::geom::Geometry*> ptrs;
    for (const geolib3::MultiPolygon2& multiPolygon : multiPolygons) {
        // return shared_ptr that is contained in geolib3::MultiPolygon2
        std::shared_ptr<const geos::geom::MultiPolygon> geosPtr
            = geolib3::internal::geolib2geosGeometry(multiPolygon);
        ptrs.push_back(
            const_cast<geos::geom::Geometry*>(
                static_cast<const geos::geom::Geometry*>(geosPtr.get())
            )
        );
    }
    std::unique_ptr<geos::geom::Geometry> united(
        geos::operation::geounion::CascadedUnion::Union(&ptrs)
    );
    return geolib3::MultiPolygon2(extractPolygons(*united));
}

double intersectionArea(const geolib3::Polygon2& polygon1,
                        const geolib3::Polygon2& polygon2) {
    geolib3::MultiPolygon2 intersection = intersectPolygons(polygon1, polygon2);

    double area = 0.;
    for (size_t i = 0; i < intersection.polygonsNumber(); i++) {
        area += intersection.polygonAt(i).area();
    }
    return area;
}

double IoU(const geolib3::Polygon2& polygon1, const geolib3::Polygon2& polygon2) {
    double intersectionAreaResult = intersectionArea(polygon1, polygon2);
    if (0 == intersectionAreaResult) {
        return 0.;
    } else {
        double unionArea = polygon1.area() + polygon2.area() - intersectionAreaResult;
        return intersectionAreaResult / unionArea;
    }
}

geolib3::MultiPolygon2
differenceMultiPolygons(
    const geolib3::MultiPolygon2& multiPolygon1,
    const geolib3::MultiPolygon2& multiPolygon2)
{
    auto geosMultiPolygon1 = geolib3::internal::geolib2geosGeometry(multiPolygon1);
    auto geosMultiPolygon2 = geolib3::internal::geolib2geosGeometry(multiPolygon2);
    std::unique_ptr<geos::geom::Geometry> intersectedGeom(
        geosMultiPolygon1->difference(geosMultiPolygon2.get())
    );

    return geolib3::MultiPolygon2(extractPolygons(*intersectedGeom));
}

} //namespace autocart
} //namespace wiki
} //namespace maps

