#include <maps/wikimap/mapspro/services/autocart/libs/post_processing/include/align_angles.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/line.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/polyline.h>

#include <limits>
#include <cmath>
#include <numeric>
#include <vector>

namespace maps {
namespace wiki {
namespace autocart {

Edge::Edge(const geolib3::Point2& pt1, const geolib3::Point2& pt2) {
    pt_ = geolib3::Point2((pt1.x()+pt2.x())*0.5, (pt1.y()+pt2.y())*0.5);
    float x = pt2.x()-pt1.x();
    float y = pt2.y()-pt1.y();
    float a = atan2(y, x);
    angle_ = a < 0? 2*M_PI+a: a;
}

float Edge::getAngle() const {
    return angle_;
}

void Edge::setAngle(float angle) {
    angle_ = angle;
}

geolib3::Point2 Edge::getPoint() const {
    return pt_;
}

bool Edge::isCollinearWith(const Edge& edge) const {
    float angDelta = (this->getAngle()-edge.getAngle())/M_PI;
    return fabs(angDelta - round(angDelta)) < 0.001;
}

geolib3::Point2 Edge::getIntersectionPoint(const Edge& edge) const {
    geolib3::Line2 thisLine(this->getPoint(),
                            geolib3::Direction2(this->getAngle()));
    geolib3::Line2 otherLine(edge.getPoint(),
                             geolib3::Direction2(edge.getAngle()));

    std::vector<geolib3::Point2> intersectionPoints;
    intersectionPoints = geolib3::intersection(thisLine, otherLine);
    if (intersectionPoints.size() == 1) {
        return intersectionPoints[0];
    } else {
        float inf = std::numeric_limits<float>::infinity();
        return geolib3::Point2(inf, inf);
    }
}

namespace {

int normalizeArrayIndex(int index, int length) {
    return index >= length? index-length: (index < 0? index+length:index);
}

std::vector<Edge> getEdges(const geolib3::Polygon2& polygon) {
    size_t ptsCnt = polygon.pointsNumber();
    std::vector<Edge> edges;
    edges.reserve(ptsCnt);
    for (size_t i = 0; i < ptsCnt; i++) {
        geolib3::Point2 pt1 = polygon.pointAt(i);
        geolib3::Point2 pt2 = polygon.pointAt((i+1)%ptsCnt);
        edges.emplace_back(pt1, pt2);
    }
    return edges;
}

std::vector<geolib3::Point2> getVertices(const std::vector<Edge>& edges) {
    std::vector<geolib3::Point2> vertices;
    Edge prevEdge = edges.back();
    for (size_t i = 0; i < edges.size(); i++) {
        vertices.push_back(edges[i].getIntersectionPoint(prevEdge));
        prevEdge = edges[i];
    }
    return vertices;
}

geolib3::Polygon2 convertEdgesToPolygon(const std::vector<Edge>& edges) {
    std::vector<geolib3::Point2> ring = getVertices(edges);
    return geolib3::Polygon2(ring);
}

std::vector<Edge> simplify(const std::vector<Edge>& edges) {
    std::vector<Edge> simplifiedEdges;
    Edge prevEdge = edges.back();

    for (size_t i = 0; i < edges.size(); i++) {
        if (!prevEdge.isCollinearWith(edges[i])) {
            simplifiedEdges.push_back(edges[i]);
            prevEdge = edges[i];
        }
    }
    return simplifiedEdges;
}

bool hasSelfIntersection(const std::vector<Edge>& edges) {
    std::vector<geolib3::Point2> pts = getVertices(edges);
    pts.push_back(pts.front());

    return !geolib3::isSimple(geolib3::Polyline2(pts));
}

bool isValid(const std::vector<Edge>& edges) {
    std::vector<Edge> simplifiedEdges = simplify(edges);
    if (simplifiedEdges.size() < 3) {
        return false;
    }

    std::vector<geolib3::Point2> vertices = getVertices(simplifiedEdges);
    constexpr float inf = 1e6;

    for (const geolib3::Point2& pt : vertices) {
        if (fabs(pt.x()) > inf || fabs(pt.y()) > inf) {
            return false;
        }
    }

    return !hasSelfIntersection(simplifiedEdges);
}

float calcVerticesOffsetWeight(const std::vector<Edge>& curEdges,
                               const std::vector<Edge>& newEdges) {
    std::vector<geolib3::Point2> curVertices = getVertices(curEdges);
    std::vector<geolib3::Point2> newVertices = getVertices(newEdges);
    std::vector<float> offsetValues;

    for (size_t i = 0; i < curVertices.size(); i++) {
        int prev = normalizeArrayIndex(i-1, curVertices.size());

        if (newEdges[i].isCollinearWith(newEdges[prev])) {
            const geolib3::Point2& p1 = newEdges[i].getPoint();
            const geolib3::Point2& p2 = newEdges[prev].getPoint();
            float a = newEdges[i].getAngle();
            float offset = fabs((p2.y()-p1.y())*cos(a)-(p2.x()-p1.x())*sin(a));
            offsetValues.push_back(offset);
        } else {
            float offset = geolib3::distance(newVertices[i], curVertices[i]);
            offsetValues.push_back(offset);
        }
    }

    float totalOffset = std::accumulate(offsetValues.begin(),
                                        offsetValues.end(), 0.f);
    float averageOffset = totalOffset/offsetValues.size();
    float weightCoeff = 3;
    float verticesOffsetWeight = 0;

    for (float offset : offsetValues) {
        float weight = offset > averageOffset? weightCoeff*offset/averageOffset: 1;
        verticesOffsetWeight += offset*weight;
    }

    return verticesOffsetWeight;
}

} // anonymous namespace

std::vector<Edge> doAlignAngles(int fixedEdgeIndex, int fromVertexIndex,
                                const std::vector<Edge>& edges,
                                float snapPrecision) {
    constexpr float snapAngle = M_PI/2.f;
    constexpr float DELTA_EPS = 0.001;
    std::vector<Edge> res = edges;

    for (size_t i = 0; i < res.size(); i++) {
        bool modified = false;
        for (size_t j = 0; j < res.size(); j++) {
            int cur = normalizeArrayIndex(fromVertexIndex+j, res.size());
            int next = normalizeArrayIndex(cur+1, res.size());

            if (next != fixedEdgeIndex) {
                float edgesAngle = res[next].getAngle()-res[cur].getAngle();
                float closestAngle = round(edgesAngle/snapAngle)*snapAngle;
                float delta = fabs(edgesAngle - closestAngle);

                if (delta < snapPrecision && delta > DELTA_EPS) {
                    res[next].setAngle(res[cur].getAngle()+closestAngle);
                    modified = true;
                }
            }
        }
        if (!modified) {
            break;
        }
    }
    return res;
}

geolib3::Polygon2 alignAngles(const geolib3::Polygon2& polygon,
                              float snapPrecision) {
    std::vector<Edge> edges = getEdges(polygon);

    float minOffsetWeight = std::numeric_limits<float>::max();
    constexpr float WEIGHT_EPS = 0.001;
    std::vector<Edge> alignEdges = edges;

    for (size_t i = 0; i < edges.size(); i++) {
        for (size_t j = 0; j < edges.size(); j++) {
            std::vector<Edge> curEdges = doAlignAngles(i, j, edges, snapPrecision);
            if (isValid(curEdges)) {
                float offsetWeight = calcVerticesOffsetWeight(edges, curEdges);
                if (offsetWeight > WEIGHT_EPS && offsetWeight < minOffsetWeight) {
                    minOffsetWeight = offsetWeight;
                    alignEdges = curEdges;
                }
            }
        }
    }

    geolib3::Polygon2 result = convertEdgesToPolygon(simplify(alignEdges));
    return result;
}

} //namespace autocart
} //namespace wiki
} //namespace maps
