#include <maps/wikimap/mapspro/services/autocart/libs/post_processing/include/post_processing.h>
#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/polygon_processing.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/vector.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/convex_hull.h>
#include <maps/libs/geolib/include/segment.h>
#include <maps/libs/geolib/include/line.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <maps/libs/tile/include/const.h>
#include <maps/libs/tile/include/utils.h>

#include <maps/libs/common/include/exception.h>

#include <opencv2/opencv.hpp>

#include <vector>
#include <cmath>
#include <utility>
#include <algorithm>
#include <set>
#include <unordered_set>
#include <optional>

namespace maps::wiki::autocart {
namespace {

/**
 * @brief Calculate distance between two rectangles.
 *     Distance between two rectangles is distance between their centers.
 *
 * @param rect1 rectangle
 * @param rect2 rectangle
 *
 * @return distance between centers of rectangles
 */
float rectDistance(const geolib3::Polygon2& rect1,
                   const geolib3::Polygon2& rect2) {
    return geolib3::distance(rect1.exteriorRing().findCentroid(), rect2.exteriorRing().findCentroid());
}

std::vector<std::set<std::pair<float, int>>>
findNeighbors(const std::vector<geolib3::Polygon2>& rectangles, float maxDist) {
    std::vector<std::set<std::pair<float, int>>> neighbors(rectangles.size());
    for (size_t i = 1; i < rectangles.size(); i++) {
        for (size_t j = 0; j < i; j++) {
            float dist = rectDistance(rectangles[i], rectangles[j]);
            if (dist < maxDist) {
                neighbors[i].emplace(dist, j);
                neighbors[j].emplace(dist, i);
            }
        }
    }

    return neighbors;
}

double maxDistance(const geolib3::Segment2& seg,
                   const std::vector<geolib3::Point2>& pts) {
    double maxDist = 0;
    for (const auto& pt : pts) {
        maxDist = std::max(maxDist, geolib3::distance(pt, seg));
    }

    return maxDist;
}

std::optional<std::pair<int, double>>
getEdgeToRemoveOverlap(const geolib3::Polygon2& rect,
                       const geolib3::Polygon2& rectToIntersect) {
    auto ptsAndEdges = intersection(rect, rectToIntersect);
    const std::vector<geolib3::Point2>& pts = ptsAndEdges.first;
    const std::unordered_set<int>& edges = ptsAndEdges.second;

    int edge = -1;
    double step;
    double maxArea = 0.;

    for (const auto& index : edges) {
        if (edges.find((index + 2) % 4) == edges.end()) {
            double edgeStep = maxDistance(rect.segmentAt(index), pts);

            double width  = geolib3::length(rect.segmentAt(index));
            double height = geolib3::length(rect.segmentAt((index + 1) % 4)) - edgeStep;
            double area = width * height;
            if (area > maxArea) {
                maxArea = area;
                edge = index;
                step = edgeStep;
            }
        }
    }
    std::optional<std::pair<int, double>> edgeAndStep;
    if (edge != -1) {
        edgeAndStep = std::make_pair(edge, step);
    }
    return edgeAndStep;
}

double cos(const geolib3::Vector2& vec1, const geolib3::Vector2& vec2) {
    double dot = geolib3::innerProduct(vec1, vec2);
    return dot / (geolib3::length(vec1) * geolib3::length(vec2));
}

bool isRectangle(const geolib3::Polygon2& polygon) {
    if (polygon.pointsNumber() != 4) {
        return false;
    }

    std::vector<geolib3::Point2> pts;
    for (size_t i = 0; i < polygon.pointsNumber(); i++) {
        pts.push_back(polygon.pointAt(i));
    }

    const double EPS = 1e-3;
    for (size_t i = 0; i < pts.size(); i++) {
        geolib3::Point2 pt0 = pts[i];
        geolib3::Point2 pt1 = pts[(i + 1) % 4];
        geolib3::Point2 pt2 = pts[(i + 2) % 4];

        geolib3::Vector2 vec1 = pt1 - pt0;
        geolib3::Vector2 vec2 = pt2 - pt1;
        if (fabs(cos(vec1, vec2)) > EPS) {
            return false;
        }
    }
    return true;
}

void removeIntersections(geolib3::Polygon2& rect1, geolib3::Polygon2& rect2) {
    geolib3::Polygon2* biggerPolygon;
    geolib3::Polygon2* smallerPolygon;

    if (rect1.area() > rect2.area()) {
        biggerPolygon = &rect1;
        smallerPolygon = &rect2;
    } else {
        biggerPolygon = &rect2;
        smallerPolygon = &rect1;
    }

    std::optional<std::pair<int, double>> edgeAndStep;
    edgeAndStep = getEdgeToRemoveOverlap(*biggerPolygon, *smallerPolygon);
    if (edgeAndStep) {
        const double BIGGER_POLYGON_STEP_RATE = 0.5;
        int edge = edgeAndStep->first;
        double step = edgeAndStep->second;
        moveEdge(*biggerPolygon, edge, step * BIGGER_POLYGON_STEP_RATE);
    } else {
        return;
    }

    edgeAndStep = getEdgeToRemoveOverlap(*smallerPolygon, *biggerPolygon);
    if (edgeAndStep) {
        int edge = edgeAndStep->first;
        double step = edgeAndStep->second;
        moveEdge(*smallerPolygon, edge, step);
    } else {
        return;
    }
}

} //namespace

float getNormalizedAngle(const geolib3::Polygon2& rect) {
    const geolib3::Point2 pt0 = rect.pointAt(0);
    const geolib3::Point2 pt1 = rect.pointAt(1);
    geolib3::Vector2 vec(pt1.x() - pt0.x(), pt1.y() - pt0.y());
    float angleRadians = atan2(vec.y(), vec.x());
    float angleDegree = angleRadians * (180.f / M_PI);
    return angleDegree - floorf(angleDegree / 90.f) * 90.f;
}

std::vector<geolib3::Polygon2>
alignRectangles(const std::vector<geolib3::Polygon2>& rectangles,
                unsigned int neighborsNumber,
                float angleDeltaDegree, float distDeltaPixel,
                unsigned int iterations)
{
    for (size_t i = 0; i < rectangles.size(); i++) {
        REQUIRE(isRectangle(rectangles[i]), "Input polygons must be rectangles");
    }

    // contains neighbors indices and distances to them for each rectangle.
    // sorted by distance
    std::vector<std::set<std::pair<float, int>>> neighbors;
    neighbors = findNeighbors(rectangles, distDeltaPixel);

    std::vector<geolib3::Polygon2> alignedRects = rectangles;
    for (unsigned int iter = 0; iter < iterations; iter++) {
        std::vector<geolib3::Polygon2> iterRects(alignedRects.size());
        for (size_t i = 0; i < alignedRects.size(); i++) {
            float normalizedAngle = getNormalizedAngle(alignedRects[i]);
            std::vector<float> angleOffsets;
            for (auto neighbor = neighbors[i].begin(); neighbor != neighbors[i].end(); neighbor++) {
                if (angleOffsets.size() == neighborsNumber) {
                    break;
                } else {
                    float neighborNormalizedAngle = getNormalizedAngle(alignedRects[neighbor->second]);
                    float angleDiff = neighborNormalizedAngle - normalizedAngle;
                    if (fabs(angleDiff) < angleDeltaDegree) {
                        angleOffsets.push_back(angleDiff);
                    } else if (90.f - fabs(angleDiff) < angleDeltaDegree) {
                        if (normalizedAngle > neighborNormalizedAngle) {
                            angleOffsets.push_back(90.f + neighborNormalizedAngle - normalizedAngle);
                        } else {
                            angleOffsets.push_back(neighborNormalizedAngle - (90.f + normalizedAngle));
                        }
                    }
                }
            }
            float avgOffset = 0.f;
            for (const float& offset : angleOffsets) {
                avgOffset += offset;
            }
            avgOffset /= angleOffsets.size() + 1;
            iterRects[i] = rotatePolygon(alignedRects[i], avgOffset, alignedRects[i].exteriorRing().findCentroid());
        }
        alignedRects = iterRects;
    }

    return alignedRects;
}

geolib3::Polygon2 getBoundingRectangle(const geolib3::PointsVector& points) {
    geolib3::Polygon2 hull = geolib3::convexHull(points);
    geolib3::Point2 centroid = hull.exteriorRing().findCentroid();

    float minAngle;
    geolib3::Polygon2 minBox;
    float minArea = std::numeric_limits<float>::max();

    const float ANGLE_EPS = 2.f;
    std::vector<float> processedAngles;
    processedAngles.reserve(hull.segmentsNumber());
    for (size_t i = 0; i < hull.segmentsNumber(); i++) {
        float segmentAngle = hull.segmentAt(i).line().direction().angle() * 180.f / M_PI;

        bool hasCloseAngleProcessed = false;
        for (const auto& angle : processedAngles) {
            float angleDiff = fmod(fabs(angle - segmentAngle), 180.f);
            if (angleDiff < ANGLE_EPS) {
                hasCloseAngleProcessed = true;
            }
        }
        if (hasCloseAngleProcessed) {
            continue;
        } else {
            processedAngles.push_back(segmentAngle);
        }

        geolib3::Polygon2 rotatedPolygon = rotatePolygon(hull, -segmentAngle, centroid);
        geolib3::Polygon2 box = rotatedPolygon.boundingBox().polygon();
        float boxArea = box.area();
        if (boxArea < minArea) {
            minAngle = segmentAngle;
            minBox = box;
            minArea = boxArea;
        }
    }

    return rotatePolygon(minBox, minAngle, centroid);
}

geolib3::Polygon2 getBoundingRectangle(const geolib3::Polygon2& polygon) {
    std::vector<geolib3::Point2> points;
    for (size_t i = 0; i < polygon.pointsNumber(); i++) {
        points.push_back(polygon.pointAt(i));
    }
    return getBoundingRectangle(points);
}

void moveEdge(geolib3::Polygon2& rect, int index, double step) {
    const size_t ptsCnt = rect.pointsNumber();
    std::vector<geolib3::Point2> pts;
    pts.reserve(ptsCnt);

    for (size_t i = 0; i < ptsCnt; i++) {
        pts.push_back(rect.pointAt(i));
    }

    geolib3::Point2& startPt = pts[index];
    geolib3::Point2& endPt = pts[(index + 1) % ptsCnt];
    const geolib3::Point2& nextPt = pts[(index + 2) % ptsCnt];

    geolib3::Vector2 shift(nextPt.x() - endPt.x(), nextPt.y() - endPt.y());
    shift.normalize();
    shift *= step;

    startPt += shift;
    endPt += shift;

    rect =  geolib3::Polygon2(pts);
}

void removeEqualPoints(std::vector<geolib3::Point2>& pts) {
    for (int i = (int)pts.size() - 1; i >= 0; i--) {
        for (int j = 0; j < i; j++) {
            if (pts[i] == pts[j]) {
                pts.erase(pts.begin() + i);
                break;
            }
        }
    }
}

std::pair<std::vector<geolib3::Point2>, std::unordered_set<int>>
intersection(const geolib3::Polygon2& polygon,
             const geolib3::Polygon2& polygonToIntersect) {
    std::vector<geolib3::Point2> pts;
    std::unordered_set<int> edgeIndices;

    // points on edges of polygons
    for (size_t i = 0; i < polygon.segmentsNumber(); i++) {
        for (size_t j = 0; j < polygonToIntersect.segmentsNumber(); j++) {
            geolib3::Segment2 seg1 = polygon.segmentAt(i);
            geolib3::Segment2 seg2 = polygonToIntersect.segmentAt(j);
            std::vector<geolib3::Point2> ptsOnEdges;
            ptsOnEdges = geolib3::intersection(seg1, seg2);
            if (ptsOnEdges.size() > 0) {
                pts.insert(pts.end(), ptsOnEdges.begin(), ptsOnEdges.end());
                edgeIndices.insert(i);
            }
        }
    }

    // points inside polygons
    for (size_t i = 0; i < polygon.segmentsNumber(); i++) {
        for (size_t j = 0; j < 2; j++) {
            geolib3::Segment2 seg = polygon.segmentAt(i);
            if(geolib3::spatialRelation(polygonToIntersect, seg[j],
                                        geolib3::SpatialRelation::Contains)){
                pts.push_back(seg[j]);
                edgeIndices.insert(i);
            }
        }
    }

    // points inside polygons
    for (size_t i = 0; i < polygonToIntersect.segmentsNumber(); i++) {
        for (size_t j = 0; j < 2; j++) {
            geolib3::Segment2 seg = polygonToIntersect.segmentAt(i);
            if(geolib3::spatialRelation(polygon, seg[j],
                                        geolib3::SpatialRelation::Contains)){
                pts.push_back(seg[j]);
            }
        }
    }

    removeEqualPoints(pts);

    return std::make_pair(pts, edgeIndices);
}

void removeIntersections(std::vector<geolib3::Polygon2>& rectangles) {
    for (size_t i = 0; i < rectangles.size(); i++) {
        REQUIRE(isRectangle(rectangles[i]), "Input polygons must be rectangles");
    }
    geolib3::StaticGeometrySearcher<geolib3::Polygon2, size_t> searcher;
    for (size_t i = 0; i < rectangles.size(); i++) {
        searcher.insert(&rectangles[i], i);
    }
    searcher.build();

    for(size_t i = 0; i < rectangles.size(); i++) {
        auto searchResult = searcher.find(rectangles[i].boundingBox());
        for (auto it = searchResult.first; it != searchResult.second; it++) {
            size_t rectangleIndx = it->value();
            if (rectangleIndx <= i) {
                continue;
            }
            removeIntersections(rectangles[i], rectangles[rectangleIndx]);
        }
    }
}

geolib3::Point2 getDisplayOrigin(const geolib3::BoundingBox& bbox, size_t zoom) {
    tile::DisplayCoord pt1 = tile::mercatorToDisplay(bbox.lowerCorner(), zoom);
    tile::DisplayCoord pt2 = tile::mercatorToDisplay(bbox.upperCorner(), zoom);
    return geolib3::Point2(std::min(pt1.x(), pt2.x()), std::min(pt1.y(), pt2.y()));
}

geolib3::Point2 imageToMercator(const geolib3::Point2& point,
                                const geolib3::Point2& origin,
                                size_t zoom) {
    return tile::displayToMercator(
        tile::DisplayCoord(origin.x() + point.x(),origin.y() + point.y()),
        zoom
    );
}

} // namespace maps::wiki::autocart
