#pragma once
#include <maps/libs/geolib/include/polygon.h>

#include <vector>
#include <optional>

namespace maps {
namespace wiki {
namespace autocart {

struct AlignAlongRoadsParams {
    AlignAlongRoadsParams()
        : maxDistToRoad(40.),
          maxConnectedRoadDist(5.),
          maxLineBendingPercent(0.1),
          maxAngleRotationDegree(45.),
          maxLineBendingAngleDegree(20.),
          alignEdgeAngleEpsDegree(5.)
    {}

    // максильное расстояние до дороги,
    // по которой может быть выполнено выравнивание.
    double maxDistToRoad;

    // максимальное расстояние между концами ломанных линий,
    // которые могут быть объединены в одну дорогу.
    double maxConnectedRoadDist;

    // максимально возможное отклонение точки в ломанной линии от прямой,
    // проходящей через ее концы. измеряется в долях от длины ломанной линии.
    double maxLineBendingPercent;

    // максимально возможный угол поворота у здания. если предлагаемый угол
    // поворота больше этой величины, то поворот не выполняется.
    double maxAngleRotationDegree;

    // максимально возможный угол между двумя прямыми, проходящими через концы
    // ломанных линий. если угол между прямыми больше этого значения,
    // то ломанные линии не будут объединяться.
    double maxLineBendingAngleDegree;

    // если у здание есть ребро, угол поворота которого отличается от угла
    // поворота дороги не более, чем на это значение,
    // то здание не поворачивается
    double alignEdgeAngleEpsDegree;
};

/**
 * @brief Aligns building polygons to roads
 *
 * @param blds   building polygons
 * @param roads  road polylines
 *
 * @return       container with aligned building polygons
 */
std::vector<geolib3::Polygon2>
alignAlongRoads(const std::vector<geolib3::Polygon2>& blds,
                const std::vector<geolib3::Polyline2>& roads,
                const AlignAlongRoadsParams& params);

/**
 * @brief Projects polygon to segment.
 * @param polygon  projected polygon
 * @param segment  segment on which polygon is projected
 * @return result of projection if it exists, otherwise return std::nullopt
 */
std::optional<geolib3::Segment2> project(const geolib3::Polygon2& polygon,
                                         const geolib3::Segment2& segment);

/**
 * @brief Connects segments to straight polylines.
 *     Segments are connected if their ends and angles of rotation are close.
 * @param segments  vector of segments
 * @param params    connection parameters
 * @return vector of connected polylines
 */
std::vector<geolib3::Polyline2>
connectSegments(const std::vector<geolib3::Segment2>& segments,
                const AlignAlongRoadsParams& params);

/**
 * @brief Checks that polyline is straight.
 *     Polyline is straight if its points have maximum distance to line passing
 *     through ends of polyline less than epsilon.
 * @param polyline  polyline to check
 * @param epsilon   maximum distance from line
 * @return true if line is straight, otherwise return false
 */
bool isStraightLine(const geolib3::Polyline2& polyline, double epsilon);

} //namespace autocart
} //namespace wiki
} //namespace maps
