#pragma once
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/polyline.h>

#include <opencv2/opencv.hpp>

#include <vector>
#include <optional>

namespace maps {
namespace wiki {
namespace autocart {

enum Direction {
    RIGHT = 0,
    UP = 1,
    LEFT = 2,
    DOWN = 3
};

// Available direction in grid polygon regularization
const std::vector<Direction> AVAILABLE_DIRECTIONS = {RIGHT, UP, LEFT, DOWN};

/**
 * @brief Checks that angle between two direction is 180 degrees.
 * @param direction1 - first direction
 * @param direction2 - second direction
 * @return true if angle between direction is equal to 180 degrees,
 *     otherwise false
 */
bool isOppositeDirections(const Direction& direction1,
                          const Direction& direction2);

/**
 * @brief The position of the point on a rectangular grid.
 *     The position (0, 0) corresponds to the origin.
 */
struct GridPoint {
    GridPoint(int col, int row)
        : col(col),
          row(row) {}
    int col, row;
};

/**
 * @brief Checks if polyline constructed by grid points has self-intersections.
 * @param gridPoints set of grid points
 * @return true, if polyline has self-intersections, otherwise false
 */
bool isSimple(const std::vector<GridPoint>& gridPoints);

/**
 * @brief Set of grid points around point in space
 */
class GridPointsSet {
public:
    /**
     * @brief Selects points from rectangular grid, distance to which from
     *     given points is less or equal than threshold.
     * @param srcPoint  - point in space
     * @param tolerance - maximum distance to grid point
     * @param gridSide  - size of rectangular grid
     */
    GridPointsSet(const geolib3::Point2& srcPoint,
                  double tolerance,
                  double gridSide);

    /**
     * @brief Counts number of points in the set.
     * @return points number
     */
    size_t pointsNumber();

    /**
     * @brief Returns a point with a given index.
     *     Index of point increases from left to right from left upper point
     *     to lower right point.
     * @param i - index of point
     * @return position of grid point
     */
    GridPoint pointAt(const size_t& index);

private:
    int minRow_;
    int minCol_;
    int maxRow_;
    int maxCol_;
};

/**
 * @brief Rotates the points counter-clockwise around the origin
 * @param points       - set of points
 * @param angleDegree  - angle of rotation in degrees
 * @return rotated points
 */
geolib3::PointsVector
rotatePoints(const geolib3::PointsVector& points, double angleDegree);

/**
 * @brief Simplifies boundary of polygon by Douglas–Peucker algorithm.
 *     https://en.wikipedia.org/wiki/Ramer–Douglas–Peucker_algorithm
 * @param points  - set of border points
 * @param epsilon - distance from line in Douglas–Peucker algorithm
 * @return points of simplified border
 */
geolib3::PointsVector
simplifyBoundary(const geolib3::PointsVector& points, double epsilon);

/**
 * @brief Removes extra points lying on one straight line segment
 *     and leaves only endpoints.
 * @param gridPoints - position of points on rectangular grid
 * @return cleared set of grid points
 */
std::vector<GridPoint>
removeExtraPolygonPoints(const std::vector<GridPoint>& gridPoints);

/**
 * @brief Checks that direction from first point to second point coincides
 *     with given direction.
 * @param pt1 - first grid point
 * @param pt2 - second grid point
 * @param direction - direction to be checked
 * @return true if direction between points is equal to given direction,
 *     otherwise false
 */
bool isCorrectDirection(const GridPoint& pt1, const GridPoint& pt2,
                        const Direction& direction);

/**
 * @brief Converts points in rectangular grid to points that form polygon
 *     boundary. Removes extra points that lie on one straight line.
 * @param gridPoints - points in rectangular grid
 * @param gridSide   - size of rectangular grid
 * @return points of polygon boundary
 */
geolib3::PointsVector
convertToPolygon(const std::vector<GridPoint>& gridPoints, double gridSide);

/**
 * @brief Regularizes given polygon.
 *     Tries to find best orthogonal polygon that close to original. Polygon is
 *     the best if it has minimum number of points and minimum deviation. Points
 *     of constructed polygon should lie on nodes of rectangular grid with given
 *     size.
 *     Read more in: https://arxiv.org/pdf/1504.06584.pdf
 * @param points    - points of unregularized boundary
 * @param tolerance - maximum distance to grid point
 * @param gridSide  - size of rectangular grid
 * @return points of regularized boundary of polygon if it is found, otherwise
 *     rotated bounding box with minimum area
 */
std::vector<cv::Point2f>
gridRegularizePolygon(const std::vector<cv::Point2f>& points,
                      double tolerance, double gridSize);

/**
 * @brief Project point on line that passes through given segment.
 * @param segment -  segment through which line passes
 * @param point   -  point for projection
 * @param alpha   -  projection coefficient.
 *     projection = alpha * end + (1 - alpha) * start
 * @return projection point
 */
geolib3::Point2 project(const geolib3::Segment2& segment,
                        const geolib3::Point2& point,
                        double* alpha);

/**
 * @brief Closes boundary of polygon and forms orthogonal polygon.
 * @param points - points of open boundary
 * @return closed boundary points
 */
std::vector<geolib3::Point2>
closeBoundary(const std::vector<geolib3::Point2>& points);

/**
 * @brief Regularizes given polygon.
 *     Tries to find orthogonal polygon that close to original:
 *     1) Simplify polygon boundary with Douglas–Peucker algorithm.
 *     2) Find the logest polygon edge. This edge is first edge of regularized
 *        polygon.
 *     3) Sequentially project points on previous edges and add projection to
 *        boundary. Original point add to boundary if distance to projection
 *        greater than given threshold.
 *     4) Close boundary.
 * @param points     - points of unregularized boundary
 * @param wallLength - minimal length of polygon edge (except last edge)
 * @return points of regularized boundary of polygon if it is found, otherwise
 *     rotated bounding box with minimum area
 */
std::vector<cv::Point2f>
projectionRegularizePolygon(const std::vector<cv::Point2f>& points,
                            double wallLength);

} //namespace autocart
} //namespace wiki
} //namespace maps
