#pragma once
#include <maps/libs/geolib/include/polygon.h>
#include <opencv2/opencv.hpp>
#include <vector>
#include <unordered_set>

namespace maps {
namespace wiki {
namespace autocart {

/**
 * @brief Return angle of rotation in [0, 90]
 *
 * @param rect  points of rectangle
 *
 * @return      angle of rotation
 */
float getNormalizedAngle(const geolib3::Polygon2& rect);

/**
 * @brief Align angles between rectangles.
 *
 * @param rects             vector with rectangles
 * @param neighborsNumber   maximum number of neighbors
 * @param angleDeltaDegree  maximum difference between neighbors angles
 * @param distDeltaPixel    maximum distance between neighbors
 * @param iterations        number of averaging operations
 * @return                  processed rectangles
 */
std::vector<geolib3::Polygon2>
alignRectangles(const std::vector<geolib3::Polygon2>& rectangles,
                unsigned int neighborsNumber,
                float angleDeltaDegree, float distDeltaPixel,
                unsigned int iterations);

/**
 * @brief Finds rectangle with minimum area that contains polygon
 *
 * @param polygon  polygon for which you want to find bounding box
 *
 * @return         rotated bounding box
 */
geolib3::Polygon2 getBoundingRectangle(const geolib3::Polygon2& polygon);

/**
 * @brief Finds rectangle with minimum area that contains set of points
 *
 * @param points  set of points
 *
 * @return        rotated bounding box
 */
geolib3::Polygon2 getBoundingRectangle(const geolib3::PointsVector& points);

/**
 * @brief Changes width or height of rectangle by moving one edge.
 *
 * @param rect   rectangle
 * @param index  index of edge to move
 * @param step   moving step
 */
void moveEdge(geolib3::Polygon2& rect, int index, double step);

/*
 * @brief Intersects polygon with another polygon.
 *    Finds points and edges forming polygon at intersection.
 *
 * @param polygon             main polygon
 * @param polygonToIntersect  polygon with which main polygon intersects
 *
 * @return pair of intersection points and indices of polygon edges adjacent to intersection area
 */
std::pair<std::vector<geolib3::Point2>, std::unordered_set<int>>
intersection(const geolib3::Polygon2& polygon,
             const geolib3::Polygon2& polygonToIntersect);

/**
 * @brief Removes intersection between rectangles
 *
 * @param intersectedRectangles  vector of rectanlges with intersections
 */
void removeIntersections(std::vector<geolib3::Polygon2>& intersectedRectangles);

/**
 * @brief Calculate display coordinates for origin of bounding box in mercator coordinates
 *
 * @param bbox  bounding box in mercator coordinates
 * @param zoom  zoom of satellite image
 *
 * @return origin of bounding box in display coordinates
 */
geolib3::Point2 getDisplayOrigin(const geolib3::BoundingBox& bbox, size_t zoom);

/**
 * @brief Convert point from image to mercator coordinates
 *
 * @param point   point in image coordinates
 * @param origin  origin of image bounding box in display coordinates
 * @param zoom    zoom of satellite image
 *
 * @return point in mercator coordinates
 */
geolib3::Point2 imageToMercator(const geolib3::Point2& point,
                                const geolib3::Point2& origin,
                                size_t zoom);
} //namespace autocart
} //namespace wiki
} //namespace maps
