#pragma once

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/xml/include/xml.h>

#include <opencv2/opencv.hpp>

#include <string>
#include <iostream>
#include <optional>

namespace maps {
namespace wiki {
namespace autocart {

const std::string SAT_PARAMS_URL = "http://core-factory-back.maps.yandex.net/satrep/ListItemsForPoint?";

struct SatelliteAngles {
    SatelliteAngles(double elev, double azimAngle)
        : elev(elev),
          azimAngle(azimAngle) {}
    // angle between horizon and direction to satellite (in radians)
    double elev;
    // clockwise angle between the direction to the south
    // and the direction of the satellite (in radians)
    double azimAngle;
};

struct PixelSize {
    // width of area that one pixel describes in satellite image
    double widthInMeters;
    // height of area
    double heightInMeters;
};

struct SatelliteParams {
    std::optional<SatelliteAngles> angles;
    PixelSize pixelSize;
};

/**
 * @brief Reads satellite angles from stream.
 *      Use for parsing xml node.
 *      Input string example:
 *          {"elev": "90", "azim_angle": "180"}
 *
 * @param in      - input stream
 * @param params  - satellite angles
 * @return input stream
 */
std::istream& operator>>(std::istream& in, std::optional<SatelliteAngles>& params);

/**
 * @brief Parses xml file containing information about satellite angles.
 *      Values are taken from node have maximum zorder and release-status
 *      "production".
 *      Input xml doc example:
 *          <?xml version='1.0' encoding='utf-8'?>
 *          <mosaics lat="50" lon="50">
 *            <mosaic zorder="1" release-status="production">
 *              <metadata>{"elev": "90", "azim_angle": "180"}</metadata>
 *            </mosaic>
 *          </mosaics>
 *
 * @param satXml - xml file
 * @return satellite angles
 */
std::optional<SatelliteAngles> parseSatXml(const xml3::Doc& satXml);

/**
 * @brief Receives information about the satellite image.
 * @param imageSize         - size of satellite image
 * @param geodeticBbox  - bbox of region in geodetic coordinates
 * @param satParamsUrl  - url to receive information about satellite angles
 * @return satellite params
 */
SatelliteParams
getSatelliteParams(const cv::Size& imageSize,
                   const geolib3::BoundingBox& geodeticBbox,
                   const std::string& satParamsUrl = SAT_PARAMS_URL);

/**
 * @brief Calculates distance in meters between two geodetic points.
 *      Distance is distance on sphere multiplied by mean radius of the Earth.
 *      Distance on sphere: https://en.wikipedia.org/wiki/Great-circle_distance
 * @param geodeticPoint1 - point in geodetic coordinates.
 *      x - lon in degrees, y - lat in degrees
 * @param geodeticPoint2 - point in geodetic coordinates
 * @return distance in meters
 */
double distanceInMeters(const geolib3::Point2& geodeticPoint1,
                        const geolib3::Point2& geodeticPoint2);

} // namespace autocart
} // namespace wiki
} // namespace maps
