#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/assessors/include/utils.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/assessors/include/assessor.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/assessors/include/statistics.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/assessors/include/utils.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/ft_type_id.h>

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_client/include/yandex/maps/mrc/toloka_client/client.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/file_utils.h>

#include <util/folder/tempdir.h>
#include <util/string/split.h>
#include <util/string/join.h>

#include <library/cpp/resource/resource.h>

#include <map>
#include <chrono>
#include <vector>
#include <fstream>
#include <filesystem>

namespace fs = std::filesystem;
using namespace maps;
using namespace maps::mrc::toloka::io;
using namespace maps::wiki::autocart::pipeline;

namespace {

static const std::string SCHEMA = "https://";
static const size_t MAX_REQUEST_ATTEMPTS = 5;
static const std::chrono::seconds TIMEOUT(10);
static const std::chrono::seconds RETRY_INITIAL_TIMEOUT(1);
static const double RETRY_TIMEOUT_BACKOFF = 2.;

static const std::string SOLUTIONS = "{ SOLUTIONS }";
static const std::string FT_TYPE_ID_TO_RU_NAME = "{ FT_TYPE_ID_TO_RU_NAME }";

static const std::string REAL_VIEWER_TEMPLATE
    = NResource::Find("resfs/file/template/real.html");
static const std::string GOLDEN_VIEWER_TEMPLATE
    = NResource::Find("resfs/file/template/golden.html");

std::vector<Assessor> getAssessors(
    const std::string& logins,
    const std::string& wikiTableName,
    const std::string& wikiOAuthToken)
{
    std::vector<Assessor> assessors
        = loadAssessorsFromWikiTable(wikiTableName, wikiOAuthToken);
    std::set<std::string> loginsSet;
    for (const std::string& login
            : StringSplitter(logins).Split(',').ToList<std::string>()) {
        loginsSet.insert(strip(login));
    }
    assessors.erase(
        std::remove_if(
            assessors.begin(), assessors.end(),
            [&](const Assessor& assessor) {
                return loginsSet.erase(assessor.login) == 0;
            }
        ),
        assessors.end()
    );
    REQUIRE(
        loginsSet.empty(),
        "Failed find assessors with logins: "
        << JoinRange(", ", loginsSet.begin(), loginsSet.end())
    );
    return assessors;
}

std::string getFTTypeIdToRussianNameJSMap() {
    std::stringstream ss;
    ss << "{";
    for (size_t i = 0; i < ALL_FT_TYPE_ID.size(); i++) {
        const FTTypeId& ftTypeId = ALL_FT_TYPE_ID[i];
        ss << encodeFTTypeId(ftTypeId) << " : '" + ruNameFTTypeId(ftTypeId) + "'";
        if (i != ALL_FT_TYPE_ID.size() - 1) {
            ss << ", ";
        };
    }
    ss << "}";
    return ss.str();
}

std::string makeRealSolutionsJSArray(
    const std::vector<AssessorTaskSolution>& solutions)
{
    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ArrayBuilder b) {
        for (const AssessorTaskSolution& solution : solutions) {
            b << [&](json::ObjectBuilder b) {
                b["bld_url"] = solution.task.bldURL;
                b["map_url"] = solution.task.mapURL;
                b["state"] = toString(solution.answer.state);
                if (TolokaState::Yes == solution.answer.state) {
                    b["height"] = solution.answer.height;
                    b["ft_type_id"] = solution.answer.ftTypeId;
                }
            };
        }
    };
    return ss.str();
}

std::string makeGoldenSolutionsJSArray(
    const std::vector<AssessorTaskSolution>& solutions)
{
    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ArrayBuilder b) {
        for (const AssessorTaskSolution& solution : solutions) {
            b << [&](json::ObjectBuilder b) {
                b["bld_url"] = solution.task.bldURL;
                b["map_url"] = solution.task.mapURL;
                b["state"] = toString(solution.answer.state);
                if (TolokaState::Yes == solution.answer.state) {
                    b["height"] = solution.answer.height;
                    b["ft_type_id"] = solution.answer.ftTypeId;
                }
                bool isCorrect = (solution.answer == solution.golden);
                b["correct"] = isCorrect;
                if (!isCorrect) {
                    b["golden_state"] = toString(solution.golden->state);
                    if (TolokaState::Yes == solution.golden->state) {
                        std::stringstream heightSS;
                        heightSS << solution.golden->heightRange.min
                                 << "-"
                                 << solution.golden->heightRange.max;
                        b["golden_height"] = heightSS.str();
                        b["golden_ft_type_ids"] << [&](json::ArrayBuilder b) {
                            for (const int& ftTypeId : solution.golden->ftTypeList.ids) {
                                b << ftTypeId;
                            }
                        };
                    }
                }
            };
        }
    };
    return ss.str();
}

std::string compileTemplate(
    std::string viewerTemplate,
    const std::map<std::string, std::string>& values)
{
    for (const auto& [key, value] : values) {
        size_t pos = viewerTemplate.find(key);
        REQUIRE(pos != std::string::npos, "Invalid config template");
        viewerTemplate.replace(pos, key.size(), value);
    }
    return viewerTemplate;
}

void writeRealViewer(
    const std::string& path,
    const std::vector<AssessorTaskSolution>& solutions)
{
    std::ofstream ofs(path);
    REQUIRE(ofs.is_open(), "Failed to create file: " + path);
    ofs << compileTemplate(
        REAL_VIEWER_TEMPLATE,
        {
            {FT_TYPE_ID_TO_RU_NAME, getFTTypeIdToRussianNameJSMap()},
            {SOLUTIONS, makeRealSolutionsJSArray(solutions)}
        }
    );
    ofs.close();
}

void writeGoldenViewer(
    const std::string& path,
    const std::vector<AssessorTaskSolution>& solutions)
{
    std::ofstream ofs(path);
    REQUIRE(ofs.is_open(), "Failed to create file: " + path);
    ofs << compileTemplate(
        GOLDEN_VIEWER_TEMPLATE,
        {
            {FT_TYPE_ID_TO_RU_NAME, getFTTypeIdToRussianNameJSMap()},
            {SOLUTIONS, makeGoldenSolutionsJSArray(solutions)}
        }
    );
    ofs.close();
}

} // namespace

int main(int argc, const char** argv)
try {
    maps::cmdline::Parser parser("Create HTML viewers with assessor solutions");

    maps::cmdline::Option<std::string> logins = parser.string("logins")
        .required()
        .help("Logins in Toloka separated by comma");

    maps::cmdline::Option<std::string> wikiOAuthToken = parser.string("wiki_token")
        .required()
        .help("Token for wiki. See https://auth.yandex-team.ru");

    maps::cmdline::Option<std::string> wikiTableName = parser.string("wiki_table")
        .required()
        .help("Name of wiki table. Example:\n"
              " * URL - https://wiki.yandex-team.ru/users/dolotov-e/Arkadija/\n"
              " * name - users/dolotov-e/Arkadija");

    maps::cmdline::Option<std::string> tolokaHost = parser.string("toloka_host")
        .required()
        .help("Toloka host. Example: toloka.yandex.ru (without http:// !)");

    maps::cmdline::Option<std::string> tolokaOAuthToken = parser.string("toloka_token")
        .required()
        .help("Token for Toloka");

    maps::cmdline::Option<std::string> projectId = parser.string("project_id")
        .required()
        .help("Toloka project id");

    maps::cmdline::Option<std::string> startDate = parser.string("start_date")
        .required()
        .help("Count tasks which was accepted not earlier than start date");

    maps::cmdline::Option<std::string> endDate = parser.string("end_date")
        .required()
        .help("Count tasks which was accepted not later than end date");

    maps::cmdline::Option<std::string> outputPath = parser.string("output")
        .required()
        .help("Path to output zip file with solutions");

    parser.parse(argc, const_cast<char**>(argv));

    REQUIRE(isCorrectDate(startDate) && isCorrectDate(endDate),
            "Invalid date format. Correct example: 2019-10-19");
    REQUIRE(startDate <= endDate, "Invalid date range");

    INFO() << "Loading assessors list";
    std::vector<Assessor> assessors = getAssessors(logins, wikiTableName, wikiOAuthToken);
    INFO() << "Loaded " << assessors.size() << " assessors";

    INFO() << "Creating Toloka client";
    TolokaClient tolokaClient(tolokaHost, tolokaOAuthToken);
    tolokaClient.setSchema(SCHEMA)
        .setMaxRequestAttempts(MAX_REQUEST_ATTEMPTS)
        .setTimeout(TIMEOUT)
        .setRetryInitialTimeout(RETRY_INITIAL_TIMEOUT)
        .setRetryTimeoutBackoff(RETRY_TIMEOUT_BACKOFF);

    INFO() << "Creating HTML viewers with solutions";
    std::list<TTempDir> tmpDirs;
    std::vector<std::string> dirPaths;
    for (const Assessor& assessor : assessors) {
        INFO() << "Loading solutions for " << assessor.login;
        std::vector<AssessorTaskSolution> solutions
            = getAssessorSolutions(
                tolokaClient, assessor.id, projectId,
                beginOfDay(startDate), endOfDay(endDate)
            );
        INFO() << "Loaded " << solutions.size() << " solutions";
        if (solutions.empty()) {
            continue;
        }
        INFO() << "Splitting real and golden tasks";
        std::vector<AssessorTaskSolution> realSolutions;
        std::vector<AssessorTaskSolution> goldenSolutions;
        for (const AssessorTaskSolution& solution : solutions) {
            if (solution.golden.has_value()) {
                goldenSolutions.push_back(solution);
            } else {
                realSolutions.push_back(solution);
            }
        }
        solutions.clear();
        INFO() << "Loaded " << realSolutions.size() << " real solutions";
        INFO() << "Loaded " << goldenSolutions.size() << " golden solutions";

        tmpDirs.emplace_back(TString(assessor.login));
        dirPaths.push_back(assessor.login);

        if (!realSolutions.empty()) {
            std::string path = common::joinPath(assessor.login, "real.html");
            writeRealViewer(path, realSolutions);
        }
        if (!goldenSolutions.empty()) {
            std::string path = common::joinPath(assessor.login, "golden.html");
            writeGoldenViewer(path, goldenSolutions);
        }
    }
    if (!dirPaths.empty()) {
        INFO() << "Archiving all viewers: " << outputPath;
        zipDirectories(outputPath, dirPaths);
    } else {
        INFO() << "There are no completed tasks during this period";
    }

    INFO() << "Done!";

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
