#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value_repr.h>

#include <fstream>

using namespace maps;

namespace {

static const std::string POOL_ID = "poolId";

std::string getPoolId(const std::string& jsonPath) {
    return json::Value::fromFile(jsonPath)[POOL_ID].toString();
}

json::Value assignPool(const json::Value& suite, const std::string poolId) {
    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        b[POOL_ID] = poolId;
        for (const std::string& field : suite.fields()) {
            b[field] = suite[field];
        }
    };
    return json::Value::fromString(ss.str());
}

} // namespace

int main(int argc, const char** argv)
try {
    maps::cmdline::Parser parser("Assign pool for suites");

    maps::cmdline::Option<std::string> inputJsonPath = parser.string("input")
        .required()
        .help("Path to input json with suites");

    maps::cmdline::Option<std::string> outputJsonPath = parser.string("output")
        .required()
        .help("Path to output json with suites");

    maps::cmdline::Option<std::string> poolIdJsonPath = parser.string("pool_id")
        .required()
        .help("Input json file with pool id");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Loading pool id: " << poolIdJsonPath;
    std::string poolId = getPoolId(poolIdJsonPath);
    INFO() << "pool id = " << poolId;

    INFO() << "Loading suites: " << inputJsonPath;
    json::Value suites = json::Value::fromFile(inputJsonPath);
    INFO() << "Loaded " << suites.size() << " suites";

    std::ofstream ofs(outputJsonPath);
    REQUIRE(ofs.is_open(), "Failed to open file: " + outputJsonPath);
    json::Builder builder(ofs);
    builder << [&](json::ArrayBuilder b) {
        for (const json::Value& suite : suites) {
            b << assignPool(suite, poolId);
        }
    };
    ofs.close();

    INFO() << "Finishing";

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
