#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/toloka/include/states.h>

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_client/include/yandex/maps/mrc/toloka_client/client.h>

#include <chrono>
#include <string>
#include <thread>

using namespace maps;
using namespace maps::mrc::toloka::io;
using namespace maps::wiki::autocart::pipeline;

namespace {

static const std::chrono::seconds TIMEOUT(10);
static const size_t MAX_REQUEST_ATTEMPTS = 6;
static const std::chrono::seconds RETRY_INITIAL_TIMEOUT(1);
static const double RETRY_TIMEOUT_BACKOFF = 2.;

static const size_t MAX_OVERLAP = 3;

static const std::string STATE = "state";

// Extract pool id from json
// Json format:
// [
//   {
//     "poolId" : "asdf"
//   }
// ]
//
std::string getPoolId(const std::string& jsonPath) {
    static const std::string FIELD_POOL_ID = "poolId";
    json::Value value = json::Value::fromFile(jsonPath);
    REQUIRE(value.size() == 1, "There can be only one pool in json");
    return value[0][FIELD_POOL_ID].as<std::string>();
}

using TaskIdToOverlap = std::unordered_map<std::string, size_t>;

TaskIdToOverlap loadTasksOverlap(
    const TolokaClient& client, const std::string& poolId)
{
    TasksResponse response = client.getTasks(Filter().byPoolId(poolId));
    const Tasks& tasks = response.tasks();
    TaskIdToOverlap taskIdToOverlap;
    for (const Task& task : tasks) {
        if (!task.knownSolutions().empty()) {
            // it is golden task
            continue;
        }
        REQUIRE(task.overlap(), "There is task without overlap");
        taskIdToOverlap[task.id()] = *(task.overlap());
    }
    return taskIdToOverlap;
}

using TaskSolutions = std::unordered_map<TolokaState, size_t>;

using TaskIdToSolutions = std::unordered_map<std::string, TaskSolutions>;

TaskIdToSolutions getTasksSolutions(
    const TolokaClient& tolokaClient, const std::string& poolId)
{
    AssignmentsResponse response = tolokaClient.getAssignments(
        Filter()
        .byPoolId(poolId)
        .byAssignmentStatus(AssignmentStatus::Accepted)
    );
    const Assignments& assignments = response.assignments();

    TaskIdToSolutions taskIdToSolutions;
    for (const Assignment& assignment : assignments) {
        const TaskSuiteItems& tasks = assignment.tasks();
        const AssignmentSolutions& solutions = assignment.solutions();
        REQUIRE(
            tasks.size() == solutions.size(),
            "There must be equal numbers of tasks and solutions"
        );
        for (size_t i = 0; i < tasks.size(); i++) {
            if (!tasks[i].knownSolutions().empty()) {
                // it is golden task
                continue;
            }
            TolokaState state;
            fromString(solutions[i].outputValues()[STATE].as<std::string>(), state);
            taskIdToSolutions[tasks[i].id()][state]++;
        }
    }

    return taskIdToSolutions;
}


} // namespace

int main(int argc, const char** argv)
try {
    maps::cmdline::Parser parser("Implements dynamic overlap for Toloka");

    maps::cmdline::Option<std::string> poolJsonPath = parser.string("pool")
        .required()
        .help("Input json with pool id");

    maps::cmdline::Option<std::string> tolokaHost = parser.string("toloka_host")
        .required()
        .help("Toloka host: <sandbox.>toloka.yandex.ru");

    maps::cmdline::Option<std::string> tolokaToken = parser.string("toloka_token")
        .required()
        .help("Toloka token with \"OAuth\" (see: https://toloka.yandex.ru)");

    maps::cmdline::Option<size_t> recheckIntervalInMinutes = parser.size_t("recheck_interval")
        .required()
        .help("Recheck toloka pool interval (in minutes)");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Creating Toloka client: " << tolokaHost;
    TolokaClient tolokaClient(tolokaHost, tolokaToken);
    tolokaClient.setTimeout(TIMEOUT)
                .setMaxRequestAttempts(MAX_REQUEST_ATTEMPTS)
                .setRetryInitialTimeout(RETRY_INITIAL_TIMEOUT)
                .setRetryTimeoutBackoff(RETRY_TIMEOUT_BACKOFF);

    INFO() << "Reading pool id from json: " << poolJsonPath;
    std::string poolId = getPoolId(poolJsonPath);
    INFO() << "Pool id: " << poolId;

    INFO() << "Getting tasks overlap in pool: " << poolId;
    TaskIdToOverlap taskIdToOverlap = loadTasksOverlap(tolokaClient, poolId);
    INFO() << "Loaded overlap for " << taskIdToOverlap.size() << " tasks";

    INFO() << "Checking pool: " << poolId;
    std::chrono::minutes recheckInterval(recheckIntervalInMinutes);
    bool isNotPoolCompleted = true;
    while (isNotPoolCompleted) {
        isNotPoolCompleted = false;
        INFO() << "Loading assignments";
        TaskIdToSolutions taskIdToSolutions = getTasksSolutions(tolokaClient, poolId);
        INFO() << "Loaded assignments for " << taskIdToSolutions.size() << " tasks";

        for (const auto& [taskId, solutions] : taskIdToSolutions) {
            auto it = taskIdToOverlap.find(taskId);
            REQUIRE(it != taskIdToOverlap.end(), "Unknown task " + taskId);
            size_t& overlap = it->second;

            bool hasFinalState = false;
            size_t solutionsCount = 0;
            for (const auto& [state, count] : solutions) {
                if (count > MAX_OVERLAP / 2) {
                    hasFinalState = true;
                }
                solutionsCount += count;
            }

            if (overlap == solutionsCount) {
                if (!hasFinalState && overlap < MAX_OVERLAP) {
                    INFO() << "Increasing overlap for task " << taskId << ": " << overlap + 1;
                    tolokaClient.setTaskOverlap(taskId, overlap + 1);
                    overlap++;
                    // there is new assignment in pool
                    isNotPoolCompleted = true;
                }
            } else {
                // not all assignments have been submitted yet
                isNotPoolCompleted = true;
            }
        }
        if (isNotPoolCompleted) {
            // if overlap of any task has been increased
            // but pool has already been closed
            Pool pool = tolokaClient.getPool(poolId);
            if (pool.status() == PoolStatus::Closed) {
                INFO() << "Open closed pool";
                tolokaClient.openPool(poolId);
            }
            INFO() << "Waiting " << recheckInterval.count() << " minutes";
            std::this_thread::sleep_for(recheckInterval);
        }
    }

    INFO() << "Pool " << poolId << " has been completed";

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
