#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/json/include/builder.h>

#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/yt_storage.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/detection_results.h>

#include <mapreduce/yt/interface/client.h>

#include <vector>
#include <fstream>
#include <sstream>

using namespace maps::wiki::autocart::pipeline;

namespace {

std::string getPolygonString(const maps::geolib3::Polygon2& polygon) {
    std::stringstream ss;
    ss << std::setprecision(12);
    for (size_t i = 0; i < polygon.pointsNumber(); i++) {
        maps::geolib3::Point2 point = polygon.pointAt(i);
        ss << "," << point.x() << "," << point.y();
    }
    ss << "," << polygon.pointAt(0).x() << "," << polygon.pointAt(0).y();
    return ss.str();
}

std::string getBldImageURL(const Building& bld) {
    static const std::string URL_PREFIX = "https://static-maps.yandex.ru/1.x/?l=sat&pl=c:7FFF00,w:4";
    return URL_PREFIX + getPolygonString(bld.toGeodeticGeom());
}

std::string getMapImageURL(const Building& bld) {
    static const std::string URL_PREFIX = "https://static-maps.yandex.ru/1.x/?l=sat&pl=c:7FFF00,w:0";
    return URL_PREFIX + getPolygonString(bld.toGeodeticGeom());
}

void resultToTask(const DetectionResult& result, maps::json::ObjectBuilder& b) {
    static const std::string INPUT_VALUES = "inputValues";
    static const std::string BLD_IMAGE = "bld_image";
    static const std::string MAP_IMAGE = "map_image";

    b[INPUT_VALUES] = [&](maps::json::ObjectBuilder b) {
        result.bld.toJson(b);
        b[RESULT_ID] = result.id;
        b[BLD_IMAGE] = getBldImageURL(result.bld);
        b[MAP_IMAGE] = getMapImageURL(result.bld);
    };
}

} // namespace

int main(int argc, const char** argv)
try {
    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("Prepare tolokers tasks");

    maps::cmdline::Option<std::string> ytConfigPath = parser.string("yt_config")
        .required()
        .help("Path to YT config");

    maps::cmdline::Option<std::string> region = parser.string("region")
        .required()
        .help("MPRO region name");

    maps::cmdline::Option<size_t> issueId = parser.size_t("issue_id")
        .required()
        .help("issue id of satellite factory");

    maps::cmdline::Option<std::string> tasksJsonPath = parser.string("tasks")
        .required()
        .help("Path to output file with tolokers tasks");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Loading YT config: " << ytConfigPath;
    YTConfig ytConfig(ytConfigPath);

    INFO() << "Create YT client: yt::hahn";
    NYT::IClientPtr client = NYT::CreateClient("hahn");

    INFO() << "Create YT storage client: " << ytConfig.storagePath();
    YTStorageClient storage(client, TString(ytConfig.storagePath()));

    INFO() << "Loading detection results:"
           << " region - " << region << ", issue id - " << issueId;
    std::vector<DetectionResult> results;
    storage.getResults(TString(region), issueId, &results);
    INFO() << "Loaded " << results.size() << " results";

    INFO() << "Removing rejected results";
    results.erase(
        std::remove_if(
            results.begin(), results.end(),
            [](const DetectionResult& result) {
                return result.state != TolokaState::Yes;
            }
        ),
        results.end()
    );
    REQUIRE(!results.empty(), "Failed to prepare tasks with empty input");
    INFO() << results.size() << " results left";

    INFO() << "Preparing results";
    std::ofstream ofs(tasksJsonPath);
    REQUIRE(ofs.is_open(), "Failed to open file: " + tasksJsonPath);
    maps::json::Builder builder(ofs);
    builder << [&](maps::json::ArrayBuilder b) {
        for (const DetectionResult& result : results) {
            b << [&](maps::json::ObjectBuilder b) {
                resultToTask(result, b);
            };
        }
    };
    ofs.close();

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
