#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/builder.h>

#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/detection_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/tolokers_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/assessors_results.h>

#include <fstream>
#include <functional>

using namespace maps::wiki::autocart::pipeline;

namespace json = maps::json;

namespace {

template <typename Result>
TolokaState getTolokaStateTemplate(const json::Value& resultJson) {
    Result result = Result::fromJson(resultJson);
    return result.state;
}

using GetTolokaStateFunc
    = std::function<
        TolokaState (const json::Value& resultJson)>;

const std::unordered_map<std::string, GetTolokaStateFunc>
    RESULT_TYPE_TO_FUNC = {
        {DetectionResult::getName(), getTolokaStateTemplate<DetectionResult>},
        {TolokersResult::getName(), getTolokaStateTemplate<TolokersResult>},
        {AssessorsResult::getName(), getTolokaStateTemplate<AssessorsResult>}
    };

std::string getResultTypeHelpStr() {
    std::stringstream ss;
    ss << "Type of results:\n";
    size_t i = 1;
    for (const auto& [type, func] : RESULT_TYPE_TO_FUNC) {
        ss << "  " << i << ") " << type << "\n";
        i++;
    }
    return ss.str();
}

} // namespace


int main(int argc, const char** argv)
try {
    maps::cmdline::Parser parser("Split buildings by Toloka state");

    maps::cmdline::Option<std::string> inputJsonPath = parser.string("input")
        .required()
        .help("Path to input json");

    maps::cmdline::Option<std::string> type = parser.string("type")
        .required()
        .help(getResultTypeHelpStr());

    maps::cmdline::Option<std::string> acceptedJsonPath = parser.string("accepted_output")
        .required()
        .help("Path to output json with accepted buildings");

    maps::cmdline::Option<std::string> rejectedJsonPath = parser.string("rejected_output")
        .required()
        .help("Path to output json with rejected buildings");

    maps::cmdline::Option<std::string> unknownJsonPath = parser.string("unknown_output")
        .required()
        .help("Path to output json with unknown buildings");

    parser.parse(argc, const_cast<char**>(argv));

    auto funcIt = RESULT_TYPE_TO_FUNC.find(type);
    REQUIRE(funcIt != RESULT_TYPE_TO_FUNC.end(), "Unavailable result type: " + type);
    GetTolokaStateFunc getTolokaState = funcIt->second;

    json::Value inputResults = json::Value::fromFile(inputJsonPath);

    std::ofstream acceptedOfs(acceptedJsonPath);
    REQUIRE(acceptedOfs.is_open(), "Failed to open file: " + acceptedJsonPath);
    std::ofstream rejectedOfs(rejectedJsonPath);
    REQUIRE(rejectedOfs.is_open(), "Failed to open file: " + rejectedJsonPath);
    std::ofstream unknownOfs(unknownJsonPath);
    REQUIRE(unknownOfs.is_open(), "Failed to open file: " + unknownJsonPath);

    acceptedOfs << "[";
    rejectedOfs << "[";
    unknownOfs << "[";

    json::Builder acceptedBuilder(acceptedOfs);
    json::Builder rejectedBuilder(rejectedOfs);
    json::Builder unknownBuilder(unknownOfs);

    for (const json::Value& resultJson : inputResults) {
        switch(getTolokaState(resultJson)) {
            case TolokaState::Yes:
                acceptedBuilder << resultJson;
                break;
            case TolokaState::No:
                rejectedBuilder << resultJson;
                break;
            default:
                unknownBuilder << resultJson;
        }
    }

    acceptedOfs << "]";
    rejectedOfs << "]";
    unknownOfs << "]";

    acceptedOfs.close();
    rejectedOfs.close();
    unknownOfs.close();

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
