#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/toloka/include/states.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/assessors/include/statistics.h>

#include <util/string/cast.h>
#include <util/string/split.h>

#include <set>

using namespace maps::mrc::toloka::io;

namespace maps::wiki::autocart::pipeline {

namespace {

static const std::string MIN = "min";
static const std::string MAX = "max";
static const std::string STATE = "state";
static const std::string HEIGHT = "height";
static const std::string FT_TYPE_ID = "ft_type_id";
static const std::string BLD_IMAGE_URL = "bld_image";
static const std::string MAP_IMAGE_URL = "map_image";

} // namespace

AssessorTask::AssessorTask(const std::string& mapURL, const std::string& bldURL)
    : mapURL(mapURL)
    , bldURL(bldURL)
{
}

// Task input values format:
// "input_values": {
//    "bld_image": "https://yandex.ru"
//    "map_image": "https://yandex.ru"
//  }
AssessorTask::AssessorTask(const TaskSuiteItem& task)
    : mapURL(task.inputValues()[MAP_IMAGE_URL].as<std::string>())
    , bldURL(task.inputValues()[BLD_IMAGE_URL].as<std::string>())
{
}


AssessorAnswer::AssessorAnswer(const TolokaState& state, int height, int ftTypeId)
    : state(state)
    , height(height)
    , ftTypeId(ftTypeId)
{
}

// Solution answer format:
//   "output_values": {
//     "state": "yes",
//     "height": "3",
//     "ft_type_id": "101"
//   }
AssessorAnswer::AssessorAnswer(const AssignmentSolution& solution) {
    const json::Value outputValues = solution.outputValues();
    fromString(outputValues[STATE].as<std::string>(), state);
    if (TolokaState::Yes == state) {
        height = std::stoi(outputValues[HEIGHT].as<std::string>());
        ftTypeId = std::stoi(outputValues[FT_TYPE_ID].as<std::string>());
    }
}


HeightRange::HeightRange(int minHeight, int maxHeight)
    : min(minHeight)
    , max(maxHeight)
{
}

HeightRange::HeightRange(const std::string& cond) {
    std::vector<TString> range = StringSplitter(cond).Split('-').ToList<TString>();
    REQUIRE(range.size() == 1 || range.size() == 2, "Incorrect height condition: " + cond);
    if (range.size() == 2) {
        min = FromString<int>(range[0]);
        max = FromString<int>(range[1]);
    } else {
        min = FromString<int>(range[0]);
        max = FromString<int>(range[0]);
    }
}

FTTypeList::FTTypeList(const std::set<int>& ftTypeIds)
    : ids(ftTypeIds)
{
}

FTTypeList::FTTypeList(const std::string& cond) {
    std::vector<TString> items = StringSplitter(cond).Split(',').ToList<TString>();
    REQUIRE(!items.empty(), "FT type list can not be empty");
    for (const TString& item : items) {
        ids.insert(FromString<int>(item));
    }
}

// Known solutions format for golden task:
//   "known_solutions" : [
//     {
//       "correctness_weight":1,
//       "output_values": {
//         "state": "yes",
//         "height": "3-6",
//         "ft_type_id": "102,108"
//       }
//     }
//   ]
AssessorGoldenAnswer::AssessorGoldenAnswer(const KnownSolutions& solutions) {
    REQUIRE(solutions.size() == 1u, "There can be only one known solution");
    const json::Value outputValues = solutions[0].outputValues();
    fromString(outputValues[STATE].as<std::string>(), state);
    if (TolokaState::Yes == state) {
        heightRange = HeightRange(outputValues[HEIGHT].as<std::string>());
        ftTypeList = FTTypeList(outputValues[FT_TYPE_ID].as<std::string>());
    }
}

AssessorGoldenAnswer::AssessorGoldenAnswer(
    const TolokaState& state,
    const HeightRange& heightRange,
    const FTTypeList& ftTypeList)
    : state(state)
    , heightRange(heightRange)
    , ftTypeList(ftTypeList)
{
}

bool isGoldenTask(const TaskSuiteItem& task) {
    return !task.knownSolutions().empty();
}

bool inRange(int value, int min, int max) {
    return min <= value && value <= max;
}

bool inSet(int value, const std::set<int>& values) {
    return values.count(value) > 0;
}

bool operator==(const AssessorAnswer& real, const AssessorGoldenAnswer& golden) {
    if (real.state != golden.state) {
        return false;
    }
    if (golden.state == TolokaState::No) {
        // do not check height and type if correct answer is "no"
        return true;
    }
    return inRange(real.height, golden.heightRange.min, golden.heightRange.max)
        && inSet(real.ftTypeId, golden.ftTypeList.ids);
}


std::unordered_map<std::string, AssessorStatistics>
getAssessorIdToStatistics(
    const TolokaClient& client,
    const std::string& projectId,
    const chrono::TimePoint& startTimePoint,
    const chrono::TimePoint& endTimePoint)
{
    std::unordered_map<std::string, AssessorStatistics> assessorIdToStatistics;
    std::unordered_map<std::string, size_t> assessorIdToCorrectAnswersCount;
    std::unordered_map<std::string, size_t> assessorIdToGoldenTasksCount;
    PoolsResponse poolsResponse = client.getPools(
        Filter().byProjectId(projectId).createdNotLaterThan(endTimePoint)
    );
    for (const Pool& pool : poolsResponse.pools()) {
        if (pool.status() != PoolStatus::Open) {
            // pool may be closed and not have last stopped time point
            // if it had never been opened
            if (!pool.lastStopped() || pool.lastStopped() < startTimePoint) {
                continue;
            }
        }
        AssignmentsResponse assignmentsResponse = client.getAssignments(
            Filter()
            .byPoolId(pool.id())
            .submittedNotEarlierThan(startTimePoint)
            .submittedNotLaterThan(endTimePoint)
            .byAssignmentStatus(AssignmentStatus::Accepted)
        );
        for (const Assignment& assignment : assignmentsResponse.assignments()) {
            std::string userId = assignment.userId();
            size_t& goldenTasksCount = assessorIdToGoldenTasksCount[userId];
            size_t& correctAnswersCount = assessorIdToCorrectAnswersCount[userId];
            for (size_t i = 0; i < assignment.tasks().size(); i++) {
                const TaskSuiteItem& task = assignment.tasks()[i];
                if (isGoldenTask(task)) {
                    AssessorAnswer answer(assignment.solutions()[i]);
                    AssessorGoldenAnswer goldenAnswer(task.knownSolutions());
                    goldenTasksCount++;
                    correctAnswersCount += (answer == goldenAnswer);
                }
            }
            assessorIdToStatistics[userId].completedTasksCount += assignment.tasks().size();
        }
    }
    for (auto& [userId, statistics] : assessorIdToStatistics) {
        size_t correctAnswerCount = assessorIdToCorrectAnswersCount[userId];
        size_t goldenTasksCount = assessorIdToGoldenTasksCount[userId];
        if (0u == goldenTasksCount) {
            statistics.quality = 1.;
        } else {
            statistics.quality = correctAnswerCount / static_cast<double>(goldenTasksCount);
        }
    }
    return assessorIdToStatistics;
}

std::vector<AssessorTaskSolution>
getAssessorSolutions(
    const TolokaClient& client,
    const std::string& assessorId,
    const std::string& projectId,
    const chrono::TimePoint& startTimePoint,
    const chrono::TimePoint& endTimePoint)
{
    std::vector<AssessorTaskSolution> solutions;
    PoolsResponse poolsResponse = client.getPools(
        Filter().byProjectId(projectId).createdNotLaterThan(endTimePoint)
    );
    for (const Pool& pool : poolsResponse.pools()) {
        if (pool.status() != PoolStatus::Open) {
            // pool may be closed and not have last stopped time point
            // if it had never been opened
            if (!pool.lastStopped() || pool.lastStopped() < startTimePoint) {
                continue;
            }
        }
        AssignmentsResponse assignmentsResponse = client.getAssignments(
            Filter()
            .byPoolId(pool.id())
            .submittedNotEarlierThan(startTimePoint)
            .submittedNotLaterThan(endTimePoint)
            .byAssignmentStatus(AssignmentStatus::Accepted)
            .byUserId(assessorId)
        );
        for (const Assignment& assignment : assignmentsResponse.assignments()) {
            for (size_t i = 0; i < assignment.tasks().size(); i++) {
                AssessorTask task(assignment.tasks()[i]);
                AssessorAnswer answer(assignment.solutions()[i]);
                if (isGoldenTask(assignment.tasks()[i])) {
                    AssessorGoldenAnswer golden(assignment.tasks()[i].knownSolutions());
                    solutions.push_back({task, answer, golden});
                } else {
                    solutions.push_back({task, answer, std::nullopt});
                }
            }
        }
    }
    return solutions;
}

} // namespace maps::wiki::autocart::pipeline
