#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/area.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/road.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/building.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/dwellplace.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/cover_by_cells.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/op_wrapper.h>

#include <maps/libs/log8/include/log8.h>

#include <maps/libs/geolib/include/const.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/hex_wkb.h>
#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/polygon_processing.h>

#include <mapreduce/yt/interface/client.h>

#include <vector>
#include <cmath>
#include <limits>
#include <algorithm>

namespace maps::wiki::autocart::pipeline {

namespace {

static const double MERCATOR_SIDE = geolib3::MERCATOR_MAX - geolib3::MERCATOR_MIN;

struct CellsRange {
    // left bottom
    Cell lb;
    // top right
    Cell tr;
};

int64_t getGridSize(double cellSizeMercator) {
    int64_t gridSize = MERCATOR_SIDE / cellSizeMercator;
    return std::max((int64_t)1, gridSize);
}

Cell normalizeCell(const Cell& cell, int64_t gridSize) {
    return Cell(
        (cell.x + gridSize) % gridSize,
        (cell.y + gridSize) % gridSize
    );
}

Cell coverPointByCell(
    const geolib3::Point2& point, double cellSizeMercator, int64_t gridSize)
{
    int64_t x = (point.x() - geolib3::MERCATOR_MIN) / cellSizeMercator;
    x = std::min(x, gridSize - 1);
    int64_t y = (point.y() - geolib3::MERCATOR_MIN) / cellSizeMercator;
    y = std::min(y, gridSize - 1);
    return {x, y};
}

CellsRange getCellsRange(
    const geolib3::BoundingBox& bbox, double cellSizeMercator, int64_t gridSize)
{
    Cell lbCell = coverPointByCell(bbox.lowerCorner(), cellSizeMercator, gridSize);
    Cell trCell = coverPointByCell(bbox.upperCorner(), cellSizeMercator, gridSize);
    return {lbCell, trCell};
}

template <class GeomType>
std::set<Cell> getIntersectedCells(
    const GeomType& mercGeom,
    double cellSizeMercator,
    const CellsRange& range)
{
    std::set<Cell> cells;
    for (int64_t x = range.lb.x; x <= range.tr.x; x++) {
        for (int64_t y = range.lb.y; y <= range.tr.y; y++) {
            Cell cell(x, y);
            if (geolib3::spatialRelation(
                    mercGeom,
                    cell.toBBox(cellSizeMercator).toMercatorGeom().polygon(),
                    geolib3::SpatialRelation::Intersects))
            {
                cells.insert(cell);
            }
        }
    }
    return cells;
}

template <class Object, class Cover>
class CoverByCellsMapper
    : public NYT::IMapper<NYT::TTableReader<NYT::TNode>,
                          NYT::TTableWriter<NYT::TNode>> {
public:
    CoverByCellsMapper() = default;
    CoverByCellsMapper(const Cover& cover)
        : cover_(cover)
    {}

    Y_SAVELOAD_JOB(cover_);

    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override {
        for (; reader->IsValid(); reader->Next()) {
            NYT::TNode row = reader->GetRow();
            Object object = Object::fromYTNode(row);

            if (!isInMercatorBBox(object)) {
                ERROR() << "Object coordinates are not valid";
                continue;
            }

            std::set<Cell> cells = cover_.coverGeomByCells(object.toMercatorGeom());
            if (!cells.empty()) {
                for (const auto& cell : cells) {
                    cell.toYTNode(row);
                    writer->AddRow(row);
                }
            }
        }
    }

private:
    bool isInMercatorBBox(const Object& object) const {
        static const geolib3::BoundingBox MERCATOR_BBOX(
            {geolib3::MERCATOR_MIN, geolib3::MERCATOR_MIN},
            {geolib3::MERCATOR_MAX, geolib3::MERCATOR_MAX}
        );
        return geolib3::spatialRelation(
            MERCATOR_BBOX, object.toMercatorGeom().boundingBox(),
            geolib3::SpatialRelation::Contains
        );
    }

    Cover cover_;
};


REGISTER_MAPPER(CoverByCellsMapper<Area, DirectlyCellsCover>);
REGISTER_MAPPER(CoverByCellsMapper<Area, AdjacentCellsCover>);

REGISTER_MAPPER(CoverByCellsMapper<Road, DirectlyCellsCover>);
REGISTER_MAPPER(CoverByCellsMapper<Road, AdjacentCellsCover>);

REGISTER_MAPPER(CoverByCellsMapper<Building, DirectlyCellsCover>);
REGISTER_MAPPER(CoverByCellsMapper<Building, AdjacentCellsCover>);

REGISTER_MAPPER(CoverByCellsMapper<Dwellplace, DirectlyCellsCover>);
REGISTER_MAPPER(CoverByCellsMapper<Dwellplace, AdjacentCellsCover>);

bool isPolygonEqualToBBox(
    const geolib3::Polygon2& polygon, const geolib3::BoundingBox& bbox)
{
    return geolib3::spatialRelation(
        polygon, bbox.polygon(),
        geolib3::SpatialRelation::Equals
    );
}

void insertCellsRange(std::set<Cell>& cells, const CellsRange& range) {
    for (int64_t x = range.lb.x; x <= range.tr.x; x++) {
        for (int64_t y = range.lb.y; y <= range.tr.y; y++) {
            cells.insert({x, y});
        }
    }
}

size_t cellsRangeSize(const CellsRange& range) {
    return (range.tr.x - range.lb.x + 1) * (range.tr.y - range.lb.y + 1);
}

std::vector<geolib3::BoundingBox> splitBBox(const geolib3::BoundingBox& bbox) {
    std::vector<geolib3::BoundingBox> bboxes;
    geolib3::Point2 center = bbox.center();
    for (const geolib3::Point2& corner : bbox.corners()) {
        bboxes.push_back({center, corner});
    }
    return bboxes;
}

} // namespace

Cell::Cell(int64_t x, int64_t y)
    : x(x),
      y(y)
{}

Cell Cell::fromYTNode(const NYT::TNode& node) {
    return Cell(node[Cell::X].AsInt64(),
                node[Cell::Y].AsInt64());
}

NYT::TNode Cell::toYTNode() const {
    NYT::TNode node;
    toYTNode(node);
    return node;
}

void Cell::toYTNode(NYT::TNode& node) const {
    node[Cell::X] = x;
    node[Cell::Y] = y;
}

BBox Cell::toBBox(double cellSizeInMercator) const {
    double minX = geolib3::MERCATOR_MIN + x * cellSizeInMercator;
    double minY = geolib3::MERCATOR_MIN + y * cellSizeInMercator;
    double maxX = minX + cellSizeInMercator;
    double maxY = minY + cellSizeInMercator;
    return BBox::fromMercatorGeom(
        geolib3::BoundingBox(
            geolib3::Point2(minX, minY),
            geolib3::Point2(maxX, maxY)
        )
    );
}

bool Cell::operator==(const Cell& other) const {
    return x == other.x && y == other.y;
}

bool Cell::operator<(const Cell& other) const {
    return x < other.x || (x == other.x && y < other.y);
}


DirectlyCellsCover::DirectlyCellsCover(double cellSizeMercator)
    : cellSizeMercator_(cellSizeMercator)
    , gridSize_(getGridSize(cellSizeMercator))
{
    REQUIRE(cellSizeMercator_ > MERCATOR_SIDE / std::numeric_limits<int64_t>::max(),
            "Cell size is too small: " + std::to_string(cellSizeMercator_));
}

template <class GeomType>
std::set<Cell> DirectlyCellsCover::coverGeomByCells(const GeomType& mercGeom) const {
    geolib3::BoundingBox mercBBox = mercGeom.boundingBox();
    CellsRange cellsRange = getCellsRange(mercBBox, cellSizeMercator_, gridSize_);
    return getIntersectedCells(mercGeom, cellSizeMercator_, cellsRange);
}

template <>
std::set<Cell> DirectlyCellsCover::coverGeomByCells<geolib3::Polygon2>(
    const geolib3::Polygon2& mercGeom) const
{
    static const size_t CELLS_RANGE_SIZE_THRESHOLD = 1000;
    std::set<Cell> cells;
    std::stack<geolib3::Polygon2> polygons({mercGeom});
    while (!polygons.empty()) {
        geolib3::Polygon2 polygon = polygons.top();
        polygons.pop();
        geolib3::BoundingBox bbox = polygon.boundingBox();
        CellsRange cellsRange = getCellsRange(bbox, cellSizeMercator_, gridSize_);
        if (isPolygonEqualToBBox(polygon, bbox)) {
            insertCellsRange(cells, cellsRange);
            continue;
        }
        if (cellsRangeSize(cellsRange) > CELLS_RANGE_SIZE_THRESHOLD) {
            for (const geolib3::BoundingBox& partBBox : splitBBox(bbox)) {
                geolib3::MultiPolygon2 intersection
                    = intersectPolygons(polygon, partBBox.polygon());
                for (size_t i = 0; i < intersection.polygonsNumber(); i++) {
                    polygons.push(intersection.polygonAt(i));
                }
            }
        } else {
            std::set<Cell> polygonCells
                = getIntersectedCells(polygon, cellSizeMercator_, cellsRange);
            cells.insert(polygonCells.begin(), polygonCells.end());
        }
    }
    return cells;
}

template <>
std::set<Cell> DirectlyCellsCover::coverGeomByCells<geolib3::MultiPolygon2>(
    const geolib3::MultiPolygon2& mercGeom) const
{
    std::set<Cell> cells;
    for (size_t i = 0; i < mercGeom.polygonsNumber(); i++) {
        std::set<Cell> polygonCells = coverGeomByCells(mercGeom.polygonAt(i));
        cells.insert(polygonCells.begin(), polygonCells.end());
    }
    return cells;
}

template
std::set<Cell> DirectlyCellsCover::coverGeomByCells(const geolib3::Point2& geom) const;

template
std::set<Cell> DirectlyCellsCover::coverGeomByCells(const geolib3::Polyline2& geom) const;


AdjacentCellsCover::AdjacentCellsCover()
    : cellSizeMercator_(DEFAULT_CELL_SIZE_MERCATOR)
    , adjacentDistanceMercator_(DEFAULT_ADJACENT_DISTANCE_MERCATOR)
    , gridSize_(getGridSize(cellSizeMercator_))
{
    validateParams();
}

AdjacentCellsCover::AdjacentCellsCover(
    double cellSizeMercator, double adjacentDistanceMercator)
    : cellSizeMercator_(cellSizeMercator)
    , adjacentDistanceMercator_(adjacentDistanceMercator)
    , gridSize_(getGridSize(cellSizeMercator_))
{
    validateParams();
}

template <class GeomType>
bool AdjacentCellsCover::isAdjacentCell(
    const Cell& cell,
    const GeomType& mercGeom) const
{
    return geolib3::spatialRelation(
        mercGeom,
        geolib3::resizeByValue(
            cell.toBBox(cellSizeMercator_).toMercatorGeom(),
            adjacentDistanceMercator_).polygon(),
        geolib3::SpatialRelation::Intersects
    );
}

void AdjacentCellsCover::validateParams() const {
    REQUIRE(cellSizeMercator_ > MERCATOR_SIDE / std::numeric_limits<int64_t>::max(),
            "Cell size is too small: " + std::to_string(cellSizeMercator_));
    REQUIRE(cellSizeMercator_ >= adjacentDistanceMercator_,
            "Cell size should be greater than adjacenet distance" +
            std::to_string(cellSizeMercator_) +
            " > " +
            std::to_string(adjacentDistanceMercator_)
    );
}

template <class GeomType>
std::set<Cell> AdjacentCellsCover::coverGeomByCells(const GeomType& mercGeom) const {
    geolib3::BoundingBox mercBBox = mercGeom.boundingBox();

    CellsRange range = getCellsRange(mercBBox, cellSizeMercator_, gridSize_);

    // get directly cells
    std::set<Cell> cells
        = getIntersectedCells(mercGeom, cellSizeMercator_, range);

    // add adjacent cells
    // top
    for (int64_t x = range.lb.x - 1; x <= range.tr.x + 1; x++) {
        Cell cell(x, range.tr.y + 1);
        if (isAdjacentCell(cell, mercGeom)) {
            cells.insert(normalizeCell(cell, gridSize_));
        }
    }
    // right
    for (int64_t y = range.lb.y - 1; y <= range.tr.y + 1; y++) {
        Cell cell(range.tr.x + 1, y);
        if (isAdjacentCell(cell, mercGeom)) {
            cells.insert(normalizeCell(cell, gridSize_));
        }
    }
    // bottom
    for (int64_t x = range.lb.x - 1; x <= range.tr.x + 1; x++) {
        Cell cell(x, range.lb.y - 1);
        if (isAdjacentCell(cell, mercGeom)) {
            cells.insert(normalizeCell(cell, gridSize_));
        }
    }
    // left
    for (int64_t y = range.lb.y - 1; y <= range.tr.y + 1; y++) {
        Cell cell(range.lb.x - 1, y);
        if (isAdjacentCell(cell, mercGeom)) {
            cells.insert(normalizeCell(cell, gridSize_));
        }
    }

    return cells;
}

template
std::set<Cell> AdjacentCellsCover::coverGeomByCells(const geolib3::Point2& geom) const;

template
std::set<Cell> AdjacentCellsCover::coverGeomByCells(const geolib3::Polygon2& geom) const;

template
std::set<Cell> AdjacentCellsCover::coverGeomByCells(const geolib3::Polyline2& geom) const;

template
std::set<Cell> AdjacentCellsCover::coverGeomByCells(const geolib3::MultiPolygon2& geom) const;


template <class Object, class CoverType>
void coverObjectsByCells(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const CoverType& cover,
    const TString& outputYTTableName)
{
    NYT::ITransactionPtr txn = client->StartTransaction();

    YTOpExecutor::MapSpec spec;
    for (const TString& inputYTTableName : inputYTTablesName) {
        spec.AddInput(inputYTTableName);
    }
    spec.AddOutput(outputYTTableName);
    YTOpExecutor::Map(
        txn,
        spec,
        new CoverByCellsMapper<Object, CoverType>(cover),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Covering objects by cells")
    );
    txn->Sort(
        NYT::TSortOperationSpec()
            .AddInput(outputYTTableName)
            .Output(outputYTTableName)
            .SortBy({Cell::X, Cell::Y})
    );

    txn->Commit();
}


template void coverObjectsByCells<Area, DirectlyCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const DirectlyCellsCover& cover,
    const TString& outputYTTableName);

template void coverObjectsByCells<Road, DirectlyCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const DirectlyCellsCover& cover,
    const TString& outputYTTableName);

template void coverObjectsByCells<Building, DirectlyCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const DirectlyCellsCover& cover,
    const TString& outputYTTableName);

template void coverObjectsByCells<Dwellplace, DirectlyCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const DirectlyCellsCover& cover,
    const TString& outputYTTableName);


template void coverObjectsByCells<Area, AdjacentCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const AdjacentCellsCover& cover,
    const TString& outputYTTableName);

template void coverObjectsByCells<Road, AdjacentCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const AdjacentCellsCover& cover,
    const TString& outputYTTableName);

template void coverObjectsByCells<Building, AdjacentCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const AdjacentCellsCover& cover,
    const TString& outputYTTableName);

template void coverObjectsByCells<Dwellplace, AdjacentCellsCover>(
    NYT::IClientBasePtr client,
    const std::vector<TString>& inputYTTablesName,
    const AdjacentCellsCover& cover,
    const TString& outputYTTableName);

} // namespace maps::wiki::autocart::pipeline
