#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/bbox.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/area.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/road.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/building.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/dwellplace.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/cover_by_cells.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/state.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/op_wrapper.h>

#include <maps/wikimap/mapspro/services/autocart/libs/utils/include/pool.h>

#include <maps/libs/log8/include/log8.h>

#include <maps/libs/geolib/include/const.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/interface/client.h>

#include <util/generic/size_literals.h>
#include <util/thread/pool.h>

#include <vector>
#include <cmath>
#include <algorithm>

namespace maps::wiki::autocart::pipeline {

namespace {

class CreateCellsByDwellplacesReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>> {
public:
    CreateCellsByDwellplacesReducer() = default;
    CreateCellsByDwellplacesReducer(double cellSizeInMercator)
        : cellSizeInMercator_(cellSizeInMercator)
    {}

    Y_SAVELOAD_JOB(cellSizeInMercator_);

    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override {
        constexpr size_t dwellplacesIndex = 0;
        constexpr size_t bldsIndex = 1;
        constexpr size_t roadsIndex = 2;
        constexpr size_t areasIndex = 3;
        if (!reader->IsValid()) {
            return;
        }

        const auto& row = reader->GetRow();

        Cell cell = Cell::fromYTNode(row);
        BBox bbox = cell.toBBox(cellSizeInMercator_);

        bool hasDwellplaces = false; // true if there is at least one place in cell

        std::vector<Building> blds;
        std::vector<Road> roads;
        std::vector<Area> areas;
        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();
            size_t tableIndex = reader->GetTableIndex();
            if (bldsIndex == tableIndex) {
                blds.push_back(Building::fromYTNode(row));
            } else if (roadsIndex == tableIndex) {
                roads.push_back(Road::fromYTNode(row));
            } else if (areasIndex == tableIndex) {
                areas.push_back(Area::fromYTNode(row));
            } else if (dwellplacesIndex == tableIndex) {
                hasDwellplaces = true;
            } else {
                throw maps::RuntimeError("Invalid table index");
            }
        }

        if (hasDwellplaces) {
            NYT::TNode node = cell.toYTNode();
            bbox.toYTNode(node);
            bldsToYTNode(blds, node);
            roadsToYTNode(roads, node);
            areasToYTNode(areas, node);
            writer->AddRow(node);
        }
    }

private:
    double cellSizeInMercator_;
};

REGISTER_REDUCER(CreateCellsByDwellplacesReducer);



class CreateCellsByRegionsReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>> {
public:
    CreateCellsByRegionsReducer() = default;
    CreateCellsByRegionsReducer(double cellSizeInMercator)
        : cellSizeInMercator_(cellSizeInMercator)
    {}

    Y_SAVELOAD_JOB(cellSizeInMercator_);

    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override {
        constexpr size_t cellsIndex = 0;
        constexpr size_t bldsIndex = 1;
        constexpr size_t roadsIndex = 2;
        constexpr size_t areasIndex = 3;
        if (!reader->IsValid()) {
            return;
        }

        const auto& row = reader->GetRow();

        Cell cell = Cell::fromYTNode(row);
        BBox bbox = cell.toBBox(cellSizeInMercator_);

        bool isIntersectedWithRegions = false;

        std::vector<Building> blds;
        std::vector<Road> roads;
        std::vector<Area> areas;
        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();
            size_t tableIndex = reader->GetTableIndex();
            if (bldsIndex == tableIndex) {
                blds.push_back(Building::fromYTNode(row));
            } else if (roadsIndex == tableIndex) {
                roads.push_back(Road::fromYTNode(row));
            } else if (areasIndex == tableIndex) {
                areas.push_back(Area::fromYTNode(row));
            } else if (cellsIndex == tableIndex) {
                isIntersectedWithRegions = true;
            } else {
                throw maps::RuntimeError("Invalid table index");
            }
        }

        if (isIntersectedWithRegions) {
            NYT::TNode node = cell.toYTNode();
            bbox.toYTNode(node);
            bldsToYTNode(blds, node);
            roadsToYTNode(roads, node);
            areasToYTNode(areas, node);
            writer->AddRow(node);
        }
    }

private:
    double cellSizeInMercator_;
};

REGISTER_REDUCER(CreateCellsByRegionsReducer);

} // namespace

void createCellsWithObjectsByDwellplaces(
    NYT::IClientBasePtr client,
    const TString& bldsYTTablePath,
    const TString& roadsYTTablePath,
    const TString& areasYTTablePath,
    const TString& dwellplacesYTTablePath,
    double cellSizeInMercator,
    double padSizeInMercator,
    const TString& outputYTTablePath)
{
    DirectlyCellsCover directlyCover(cellSizeInMercator);
    AdjacentCellsCover adjacentCover(cellSizeInMercator, padSizeInMercator);

    NYT::TTempTable dwellplacesCellsTable = State::getTempTable(client);
    NYT::TTempTable bldsCellsTable = State::getTempTable(client);
    NYT::TTempTable roadsCellsTable = State::getTempTable(client);
    NYT::TTempTable areasCellsTable = State::getTempTable(client);

    ThreadPool mtpQueue(4);
    mtpQueue.Add([&]{
        coverObjectsByCells<Dwellplace>(
            client,
            {dwellplacesYTTablePath},
            directlyCover,
            dwellplacesCellsTable.Name()
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Building>(
            client,
            {bldsYTTablePath},
            adjacentCover,
            bldsCellsTable.Name()
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Road>(
            client,
            {roadsYTTablePath},
            adjacentCover,
            roadsCellsTable.Name()
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Area>(
            client,
            {areasYTTablePath},
            adjacentCover,
            areasCellsTable.Name()
        );
    });
    // wait for completion
    mtpQueue.Wait();

    YTOpExecutor::Reduce(
        client,
        YTOpExecutor::ReduceSpec()
            .AddInput(dwellplacesCellsTable.Name())
            .AddInput(bldsCellsTable.Name())
            .AddInput(roadsCellsTable.Name())
            .AddInput(areasCellsTable.Name())
            .AddOutput(outputYTTablePath)
            .ReduceBy({Cell::X, Cell::Y}),
        new CreateCellsByDwellplacesReducer(cellSizeInMercator),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Creating cells using dwellplaces")
            .MemoryLimit(2_GB)
    );
}

void createCellsWithObjectsByRegions(
    NYT::IClientBasePtr client,
    const TString& bldsYTTablePath,
    const TString& roadsYTTablePath,
    const TString& areasYTTablePath,
    const geolib3::MultiPolygon2& mercRegions,
    double cellSizeInMercator,
    double padSizeInMercator,
    const TString& outputYTTablePath)
{
    DirectlyCellsCover directlyCover(cellSizeInMercator);
    AdjacentCellsCover adjacentCover(cellSizeInMercator, padSizeInMercator);

    NYT::TTempTable cellsTable = State::getTempTable(client);
    NYT::TTempTable bldsCellsTable = State::getTempTable(client);
    NYT::TTempTable roadsCellsTable = State::getTempTable(client);
    NYT::TTempTable areasCellsTable = State::getTempTable(client);

    ThreadPool mtpQueue(4);
    mtpQueue.Add([&]{
        NYT::TTableWriterPtr<NYT::TNode> writer
            = client->CreateTableWriter<NYT::TNode>(cellsTable.Name());
        for (const Cell& cell : directlyCover.coverGeomByCells(mercRegions)) {
            writer->AddRow(cell.toYTNode());
        }
        writer->Finish();
        client->Sort(
            NYT::TSortOperationSpec()
                .AddInput(cellsTable.Name())
                .Output(cellsTable.Name())
                .SortBy({Cell::X, Cell::Y})
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Building>(
            client,
            {bldsYTTablePath},
            adjacentCover,
            bldsCellsTable.Name()
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Road>(
            client,
            {roadsYTTablePath},
            adjacentCover,
            roadsCellsTable.Name()
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Area>(
            client,
            {areasYTTablePath},
            adjacentCover,
            areasCellsTable.Name()
        );
    });
    // wait for completion
    mtpQueue.Wait();

    YTOpExecutor::Reduce(
        client,
        YTOpExecutor::ReduceSpec()
            .AddInput(cellsTable.Name())
            .AddInput(bldsCellsTable.Name())
            .AddInput(roadsCellsTable.Name())
            .AddInput(areasCellsTable.Name())
            .AddOutput(outputYTTablePath)
            .ReduceBy({Cell::X, Cell::Y}),
        new CreateCellsByRegionsReducer(cellSizeInMercator),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Creating cells using geometries of regions")
            .MemoryLimit(2_GB)
    );
}

} // namespace maps::wiki::autocart::pipeline
