#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/dwellplace.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/extract_unique_dwellplaces.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/cover_by_cells.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/op_wrapper.h>

#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/util/ypath_join.h>
#include <mapreduce/yt/interface/client.h>

namespace maps::wiki::autocart::pipeline {

namespace {

class MergeDuplicatesByCellsReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>> {
public:
    MergeDuplicatesByCellsReducer() = default;
    MergeDuplicatesByCellsReducer(double distanceInMercator)
        : distanceInMercator_(distanceInMercator)
    {}

    Y_SAVELOAD_JOB(distanceInMercator_);

    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override {
        if (!reader->IsValid()) {
            return;
        }

        NYT::TNode row = reader->GetRow();

        Cell cell = Cell::fromYTNode(row);
        double x = geolib3::MERCATOR_MIN + (cell.x + 0.5) * 2 * distanceInMercator_;
        double y = geolib3::MERCATOR_MIN + (cell.y + 0.5) * 2 * distanceInMercator_;

        Dwellplace mergedPlace = Dwellplace::fromMercatorGeom(geolib3::Point2(x, y));

        writer->AddRow(mergedPlace.toYTNode());
    }

private:
    double distanceInMercator_;
};

REGISTER_REDUCER(MergeDuplicatesByCellsReducer);

void removeDwellplacesDuplicates(
    NYT::ITransactionPtr txn,
    const std::vector<TString>& inputYTTablesName,
    double duplicateDistanceInMercator,
    const TString& outputYTTableName)
{
    DirectlyCellsCover cover(2 * duplicateDistanceInMercator);

    NYT::TTempTable cellsTable(txn);
    coverObjectsByCells<Dwellplace>(
        txn,
        inputYTTablesName,
        cover,
        cellsTable.Name()
    );

    YTOpExecutor::Reduce(
        txn,
        YTOpExecutor::ReduceSpec()
            .AddInput(cellsTable.Name())
            .AddOutput(outputYTTableName)
            .ReduceBy({Cell::X, Cell::Y}),
        new MergeDuplicatesByCellsReducer(duplicateDistanceInMercator),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Extracting unique dwellplaces")
    );
}

} // namespace

void extractUniqueDwellplaces(
    NYT::IClientBasePtr client,
    const TString& dwellplacesYTFolderName,
    const TString& outputYTTableName,
    double duplicateDistanceInMercator)
{
    static const size_t BATCH_SIZE = 10;

    NYT::ITransactionPtr txn = client->StartTransaction();

    REQUIRE(txn->Exists(dwellplacesYTFolderName), "Dwellpalces folder does not exist");
    std::vector<TString> inputTableNames;
    for (const NYT::TNode& name : txn->List(dwellplacesYTFolderName)) {
        inputTableNames.push_back(NYT::JoinYPaths(dwellplacesYTFolderName, name.AsString()));
    }
    for (size_t begin = 0; begin < inputTableNames.size(); begin += BATCH_SIZE) {
        size_t end = std::min(begin + BATCH_SIZE, inputTableNames.size());
        std::vector<TString> batchTableNames;
        for (size_t i = begin; i < end; i++) {
            batchTableNames.push_back(inputTableNames[i]);
        }
        batchTableNames.push_back(outputYTTableName);
        removeDwellplacesDuplicates(
            txn,
            batchTableNames,
            duplicateDistanceInMercator,
            outputYTTableName
        );
    }

    txn->Commit();
}

} // namespace maps::wiki::autocart::pipeline
