#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/state.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/filter_by_ft_type_id.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/op_wrapper.h>

#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/interface/client.h>

#include <util/generic/size_literals.h>

#include <set>

namespace maps::wiki::autocart::pipeline {

namespace {

static const TString FT_ID = "ft_id";
static const TString SHAPE = "shape";
static const TString FT_TYPE_ID = "ft_type_id";

class FilterByFTTypeIdMapper
    : public NYT::IMapper<NYT::TTableReader<NYT::TNode>,
                          NYT::TTableWriter<NYT::TNode>> {
public:
    FilterByFTTypeIdMapper() = default;
    FilterByFTTypeIdMapper(const std::vector<FTTypeId>& ftTypeIds) {
        for (const FTTypeId& ftTypeId : ftTypeIds) {
            codes_.insert(encodeFTTypeId(ftTypeId));
        }
    }

    Y_SAVELOAD_JOB(codes_);

    void Do(NYT::TTableReader<NYT::TNode>* reader,
                    NYT::TTableWriter<NYT::TNode>* writer) override
    {
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& row = reader->GetRow();
            int code = row[FT_TYPE_ID].AsInt64();
            if (codes_.count(code)) {
                writer->AddRow(row);
            }
        }
    }

private:
    std::set<int> codes_;
};

REGISTER_MAPPER(FilterByFTTypeIdMapper);

class MergeByFTIdReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>> {
public:
    void Do(NYT::TTableReader<NYT::TNode>* reader,
                    NYT::TTableWriter<NYT::TNode>* writer) override
    {
        std::optional<int> code;
        std::optional<TString> shape;
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& row = reader->GetRow();
            size_t tableIndex = reader->GetTableIndex();
            if (0u == tableIndex) {
                code = row[FT_TYPE_ID].AsInt64();
            } else {
                shape = row[SHAPE].AsString();
            }
        }
        if (code.has_value() && shape.has_value()) {
            writer->AddRow(
                NYT::TNode()
                (FT_TYPE_ID, code.value())
                (SHAPE, shape.value())
            );
        }
    }
};

REGISTER_REDUCER(MergeByFTIdReducer);

} // namespace

void filterAreasByFTTypeId(
    NYT::IClientBasePtr client,
    const std::vector<TString>& ftYTTableNames,
    const std::vector<TString>& ftGeomYTTableNames,
    const std::vector<FTTypeId>& ftTypeIds,
    const TString& outputYTTableName)
{
    NYT::TTempTable ftTypeTable = State::getTempTable(client);
    YTOpExecutor::MapSpec ftTypeSpec;
    for (const TString& ftYTTableName : ftYTTableNames) {
        ftTypeSpec.AddInput(ftYTTableName);
    }
    ftTypeSpec.AddOutput(ftTypeTable.Name());
    YTOpExecutor::Map(
        client,
        ftTypeSpec,
        new FilterByFTTypeIdMapper(ftTypeIds),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Filtering areas by ft_type_id")
    );

    client->Sort(
        NYT::TSortOperationSpec()
            .AddInput(ftTypeTable.Name())
            .Output(ftTypeTable.Name())
            .SortBy({FT_ID})
    );

    YTOpExecutor::ReduceSpec mergeSpec;
    mergeSpec.AddInput(ftTypeTable.Name());
    for (const TString& ftGeomYTTableName : ftGeomYTTableNames) {
        mergeSpec.AddInput(ftGeomYTTableName);
    }
    mergeSpec.AddOutput(outputYTTableName);
    mergeSpec.ReduceBy({FT_ID});
    YTOpExecutor::Reduce(
        client,
        mergeSpec,
        new MergeByFTIdReducer(),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Merging tables by ft_id")
            .MemoryLimit(1_GB)
    );
}

} // namespace maps::wiki::autocart::pipeline
