#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/road.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/building.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/dwellplace.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/inference_auto_toloker.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/op_wrapper.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/detection_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/batch.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/rows_count.h>

#include <maps/wikimap/mapspro/services/autocart/libs/auto_toloker/include/auto_toloker.h>

#include <maps/wikimap/mapspro/services/autocart/libs/satellite/include/load_sat_image.h>

#include <maps/libs/log8/include/log8.h>

#include <maps/libs/geolib/include/const.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/bounding_box.h>

#include <maps/libs/tile/include/utils.h>

#include <maps/libs/common/include/base64.h>

#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/interface/client.h>

#include <util/generic/size_literals.h>

#include <opencv2/opencv.hpp>

#include <vector>
#include <cmath>
#include <algorithm>

namespace maps::wiki::autocart::pipeline {

namespace {

cv::Mat drawBuilding(
    const geolib3::BoundingBox& bbox,
    const geolib3::Polygon2& mercatorBld,
    size_t zoom)
{
    auto firstCorner = tile::mercatorToDisplaySigned(bbox.lowerCorner(), zoom);
    auto secondCorner = tile::mercatorToDisplaySigned(bbox.upperCorner(), zoom);
    tile::SignedDisplayCoord leftTop(std::min(firstCorner.x(), secondCorner.x()),
                                     std::min(firstCorner.y(), secondCorner.y()));
    size_t maskWidth = abs(firstCorner.x() - secondCorner.x()) + 1;
    size_t maskHeight = abs(firstCorner.y() - secondCorner.y()) + 1;
    cv::Mat mask = cv::Mat::zeros(maskHeight, maskWidth, CV_8UC1);

    std::vector<cv::Point2i> polygon;
    for (size_t i = 0; i < mercatorBld.pointsNumber(); i++) {
        geolib3::Point2 point = mercatorBld.pointAt(i);
        tile::SignedDisplayCoord dispPoint = tile::mercatorToDisplaySigned(point, zoom);
        polygon.emplace_back(dispPoint.x() - leftTop.x(), dispPoint.y() - leftTop.y());
    }

    std::vector<std::vector<cv::Point2i>> polygons(1, polygon);
    cv::fillPoly(mask, polygons, cv::Scalar::all(255));

    return mask;
}


struct InputData {
    Building bld;
    cv::Mat image;
    cv::Mat mask;
};

// Inference AutoToloker for all buildings in table
// Each column should has 'hex_wkb' column
class AutoTolokerMapper
    : public NYT::IMapper<NYT::TTableReader<NYT::TNode>,
                          NYT::TTableWriter<NYT::TNode>> {
public:
    AutoTolokerMapper() = default;

    AutoTolokerMapper(
        const TString& tileSourceUrl,
        size_t zoom,
        double padRatio,
        double threshold);

    Y_SAVELOAD_JOB(tileSourceUrl_, zoom_, padRatio_, threshold_);

    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override;

private:
    InputData loadInputData(const NYT::TNode& node) const {
        InputData data;
        data.bld = Building::fromYTNode(node);
        geolib3::BoundingBox mercatorBbox = data.bld.toMercatorGeom().boundingBox();
        mercatorBbox = geolib3::resizeByRatio(mercatorBbox, 1 + padRatio_);
        data.image = loadSatImage(mercatorBbox, zoom_, tileSourceUrl_.data());
        data.mask = drawBuilding(mercatorBbox, data.bld.toMercatorGeom(), zoom_);
        return data;
    }

    TolokaState makeTolokerDecision(double confidence) const;

    TString tileSourceUrl_;
    size_t zoom_;
    double padRatio_;
    double threshold_;
};

AutoTolokerMapper::AutoTolokerMapper(
    const TString& tileSourceUrl,
    size_t zoom,
    double padRatio,
    double threshold)
    : tileSourceUrl_(tileSourceUrl),
      zoom_(zoom),
      padRatio_(padRatio),
      threshold_(threshold)
{}

TolokaState AutoTolokerMapper::makeTolokerDecision(double confidence) const {
    return confidence > threshold_ ? TolokaState::Yes : TolokaState::No;
}

void AutoTolokerMapper::Do(
    NYT::TTableReader<NYT::TNode>* reader,
    NYT::TTableWriter<NYT::TNode>* writer)
{
    AutoToloker toloker;
    for (; reader->IsValid(); reader->Next()) {
        InputData data;
        try {
            data = loadInputData(reader->GetRow());
        } catch (const std::exception& e) {
            WARN() << e.what();
            continue;
        }
        DetectionResult result;
        result.id = reader->GetRowIndex();
        result.bld = data.bld;
        result.confidence = toloker.classify(data.image, data.mask);
        result.state = makeTolokerDecision(result.confidence);
        writer->AddRow(result.toYTNode());
    }
}

REGISTER_MAPPER(AutoTolokerMapper);

} // namespace

void inferenceAutoToloker(
    NYT::IClientBasePtr client,
    const TString& inputYTTablePath,
    const AutoTolokerConfig& config,
    const TString& outputYTTablePath,
    EraseBatch mode)
{
    static const size_t MAX_JOB_COUNT = 100000;

    size_t batchSize = std::min(
        config.jobSize() * MAX_JOB_COUNT,
        getRowsCount(client, inputYTTablePath)
    );
    processBatches(
        client,
        inputYTTablePath,
        [&](NYT::IClientBasePtr batchClient,
            const NYT::TRichYPath& batchRYTPath,
            const NYT::TRichYPath& outputRYTPath)
        {
            size_t jobCount = std::max(getRowsCount(batchRYTPath) / config.jobSize(), 1ul);
            YTOpExecutor::Map(
                batchClient,
                YTOpExecutor::MapSpec()
                    .AddInput(batchRYTPath)
                    .AddOutput(outputRYTPath),
                new AutoTolokerMapper(
                    config.tileSourceURL(),
                    config.zoom(),
                    config.padRatio(),
                    config.threshold()
                ),
                YTOpExecutor::Options()
                    .Title("[Buildings detector] Validating buildings by auto toloker")
                    .UseGPU(1)
                    .JobCount(jobCount)
                    .RunningJobCount(config.runningJobCount())
                    .MemoryLimit(8_GB)
            );
        },
        outputYTTablePath,
        batchSize,
        mode
    );
}

} // namespace maps::wiki::autocart::pipeline
