#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/remove_rejected.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/state.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/detection/include/cover_by_cells.h>

#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/op_wrapper.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/strings.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/detection_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/tolokers_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/assessors_results.h>
#include <maps/wikimap/mapspro/services/autocart/libs/utils/include/pool.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/building.h>

#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/polygon_processing.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <util/generic/size_literals.h>

#include <mapreduce/yt/interface/client.h>

namespace maps::wiki::autocart::pipeline {

namespace {

const TString IS_REJECTED = "is_rejected";

class SearchRejectedReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>> {
public:
    SearchRejectedReducer() = default;
    SearchRejectedReducer(double rejectedIOUThreshold)
        : rejectedIOUThreshold_(rejectedIOUThreshold)
    {}

    Y_SAVELOAD_JOB(rejectedIOUThreshold_);

    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override {
        constexpr size_t inputIndex = 0;
        constexpr size_t tolokersIndex = 1;
        constexpr size_t assessorsIndex = 2;

        std::vector<NYT::TNode> inputBldRows;
        std::vector<geolib3::Polygon2> rejectedGeoms;

        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();
            size_t tableIndex = reader->GetTableIndex();
            if (inputIndex == tableIndex) {
                inputBldRows.push_back(row);
            } else if (tolokersIndex == tableIndex) {
                TolokersResult result = TolokersResult::fromYTNode(row);
                if (TolokaState::No == result.state) {
                    rejectedGeoms.push_back(result.bld.toMercatorGeom());
                }
            } else if (assessorsIndex == tableIndex) {
                AssessorsResult result = AssessorsResult::fromYTNode(row);
                if (TolokaState::No == result.state) {
                    rejectedGeoms.push_back(result.bld.toMercatorGeom());
                }
            } else {
                throw maps::RuntimeError("Invalid table index");
            }
        }
        geolib3::StaticGeometrySearcher<geolib3::Polygon2, geolib3::Polygon2*> searcher;
        for (size_t i = 0; i < rejectedGeoms.size(); i++) {
            searcher.insert(&(rejectedGeoms[i]), &(rejectedGeoms[i]));
        }
        searcher.build();

        for (NYT::TNode bldRow : inputBldRows) {
            geolib3::Polygon2 bldGeom = Building::fromYTNode(bldRow).toMercatorGeom();
            bool isRejected = false;
            auto searchResult = searcher.find(bldGeom.boundingBox());
            for (auto it = searchResult.first; it != searchResult.second; it++) {
                if (IoU(bldGeom, *(it->value())) > rejectedIOUThreshold_) {
                    isRejected = true;
                    break;
                }
            }
            writer->AddRow(bldRow(IS_REJECTED, isRejected));
        }
    }

private:
    double rejectedIOUThreshold_;
};

REGISTER_REDUCER(SearchRejectedReducer);

class RemoveRejectedReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>> {
public:
    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override {
        if (!reader->IsValid()) {
            return;
        }
        DetectionResult result = DetectionResult::fromYTNode(reader->GetRow());
        bool isRejected = false;
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& row = reader->GetRow();
            isRejected = isRejected || row[IS_REJECTED].AsBool();
        }
        if (!isRejected) {
            writer->AddRow(result.toYTNode());
        }
    }
};

REGISTER_REDUCER(RemoveRejectedReducer);

} // namespace

void removeRejectedBuildings(
    NYT::IClientPtr client,
    const TString& inputYTTablePath,
    const DetectorConfig& config,
    const YTStorageClient& storage,
    const TString& outputYTTablePath)
{
    NYT::TTempTable tolokersYTTable = State::getTempTable(client);
    storage.cloneResultsTable<TolokersResult>(client, tolokersYTTable.Name());
    NYT::TTempTable assessorsYTTable = State::getTempTable(client);
    storage.cloneResultsTable<AssessorsResult>(client, assessorsYTTable.Name());

    DirectlyCellsCover directlyCover(config.cellSizeMercator());
    AdjacentCellsCover adjacentCover(config.cellSizeMercator(), config.padSizeMercator());

    NYT::TTempTable inputCellsYTTable = State::getTempTable(client);
    NYT::TTempTable tolokersCellsYTTable = State::getTempTable(client);
    NYT::TTempTable assessorsCellsYTTable = State::getTempTable(client);
    ThreadPool mtpQueue(3);
    mtpQueue.Add([&]{
        coverObjectsByCells<Building>(
            client,
            {inputYTTablePath},
            directlyCover,
            inputCellsYTTable.Name()
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Building>(
            client,
            {tolokersYTTable.Name()},
            adjacentCover,
            tolokersCellsYTTable.Name()
        );
    });
    mtpQueue.Add([&]{
        coverObjectsByCells<Building>(
            client,
            {assessorsYTTable.Name()},
            adjacentCover,
            assessorsCellsYTTable.Name()
        );
    });
    // wait for completion
    mtpQueue.Wait();

    NYT::TTempTable rejectionYTTable = State::getTempTable(client);

    YTOpExecutor::Reduce(
        client,
        YTOpExecutor::ReduceSpec()
            .AddInput(inputCellsYTTable.Name())
            .AddInput(tolokersCellsYTTable.Name())
            .AddInput(assessorsCellsYTTable.Name())
            .AddOutput(rejectionYTTable.Name())
            .ReduceBy({Cell::X, Cell::Y}),
        new SearchRejectedReducer(config.rejectedIOUThreshold()),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Searching rejected buildings")
            .MemoryLimit(4_GB)
    );
    client->Sort(
        NYT::TSortOperationSpec()
            .AddInput(rejectionYTTable.Name())
            .Output(rejectionYTTable.Name())
            .SortBy({RESULT_ID})
    );
    YTOpExecutor::Reduce(
        client,
        YTOpExecutor::ReduceSpec()
            .AddInput(rejectionYTTable.Name())
            .AddOutput(outputYTTablePath)
            .ReduceBy({RESULT_ID}),
        new RemoveRejectedReducer(),
        YTOpExecutor::Options()
            .Title("[Buildings detector] Removing rejected buildings")
    );
}

} // namespace maps::wiki::autocart::pipeline
