#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/factory/include/release.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/exception.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/serialization.h>
#include <yandex/maps/geolib3/sproto.h>
#include <yandex/maps/proto/factory/mosaics.sproto.h>
#include <yandex/maps/proto/factory/release.sproto.h>
#include <yandex/maps/shell_cmd.h>
#include <maps/libs/xml/include/xml.h>

#include <contrib/libs/gdal/gcore/gdal.h>
#include <contrib/libs/gdal/ogr/ogrsf_frmts/ogrsf_frmts.h>
#include <contrib/libs/geos/include/geos/geom/Geometry.h>
#include <contrib/libs/geos/include/geos/geom/MultiPolygon.h>
#include <contrib/libs/geos/include/geos/operation/union/CascadedUnion.h>
#include <util/folder/tempdir.h>

#include <string_view>

namespace sfactory = yandex::maps::sproto::factory;

namespace maps::wiki::autocart::pipeline {

namespace {

void unzip(const std::string& zip, const std::string& path)
{
    const auto result = maps::shell::runCmd("unzip -q " + zip + " -d " + path);
    REQUIRE(result.exitCode == 0, "Failed to unzip file "
        << path << " : " << result.stdErr);
}

ReleaseGeometries readShapefile(const std::string& path) {
    static const char* FIELD_ZMIN = "zmin";
    static const char* FIELD_ZMAX = "zmax";

    GDALAllRegister();
    std::unique_ptr<GDALDataset> datasetPtr(static_cast<GDALDataset*>(
        GDALOpenEx(
            path.c_str(), (GDAL_OF_READONLY | GDAL_OF_VECTOR | GDAL_OF_VERBOSE_ERROR),
            nullptr, nullptr, nullptr
        )
    ));
    REQUIRE(datasetPtr != nullptr, "Failed to open shapefile");

    // The returned layer remains owned by the
    // GDALDataset and should not be deleted by the application.
    // See GDAL documentation - https://gdal.org
    OGRLayer* layerPtr = datasetPtr->GetLayer(0);
    REQUIRE(layerPtr != nullptr, "Failed to get layer(0) from dataset");
    REQUIRE(
        wkbFlatten(layerPtr->GetGeomType()) == wkbUnknown ||
        wkbFlatten(layerPtr->GetGeomType()) == wkbPolygon ||
        wkbFlatten(layerPtr->GetGeomType()) == wkbMultiPolygon,
        "Shapefile " + path + " does not contain polygons"
    );

    ReleaseGeometries releaseGeometries;
    for (const OGRFeatureUniquePtr& featurePtr : *layerPtr) {
        REQUIRE(featurePtr != nullptr, "Incorrect pointer to feature");
        int zminIndex = featurePtr->GetFieldIndex(FIELD_ZMIN);
        // index == -1 if no match found
        REQUIRE(zminIndex != -1, "Feature does not have field " << FIELD_ZMIN);
        int zmin = featurePtr->GetFieldAsInteger(zminIndex);
        int zmaxIndex = featurePtr->GetFieldIndex(FIELD_ZMAX);
        REQUIRE(zmaxIndex != -1, "Feature does not have field " << FIELD_ZMAX);
        int zmax = featurePtr->GetFieldAsInteger(zmaxIndex);
        // Pointer to internal feature geometry. This object should not be modified.
        OGRGeometry* geometryPtr = featurePtr->GetGeometryRef();
        REQUIRE(geometryPtr != nullptr, "Failed to get geometry of feature");
        OGRwkbGeometryType geometryType = geometryPtr->getGeometryType();
        REQUIRE(
            wkbFlatten(geometryType) == wkbPolygon ||
            wkbFlatten(geometryType) == wkbMultiPolygon,
            "Geometry should be polygon or multipolygon"
        );
        if(wkbFlatten(geometryType) == wkbPolygon) {
            OGRPolygon* polygonPtr = geometryPtr->toPolygon();
            REQUIRE(polygonPtr != nullptr, "Failed to cast geometry to polygon");
            std::vector<uint8_t> buff(polygonPtr->WkbSize());
            REQUIRE(polygonPtr->exportToWkb(OGRwkbByteOrder::wkbNDR, buff.data()) == OGRERR_NONE,
                    "Cannot export polygon to WKB.");
            releaseGeometries.push_back(
                {zmin, zmax, geolib3::MultiPolygon2({geolib3::WKB::read<geolib3::Polygon2>(buff)})}
            );
        } else if (wkbFlatten(geometryType) == wkbMultiPolygon) {
            OGRMultiPolygon* multiPolygonPtr = geometryPtr->toMultiPolygon();
            REQUIRE(multiPolygonPtr != nullptr, "Failed to cast geometry to multipolygon");
            std::vector<uint8_t> buff(multiPolygonPtr->WkbSize());
            REQUIRE(multiPolygonPtr->exportToWkb(OGRwkbByteOrder::wkbNDR, buff.data()) == OGRERR_NONE,
                    "Cannot export multipolygon to WKB.");
            releaseGeometries.push_back({zmin, zmax, geolib3::WKB::read<geolib3::MultiPolygon2>(buff)});
        }
    }
    return releaseGeometries;
}


Release
convertToRelease(const sfactory::release::Release& release)
{
    REQUIRE(release.issueId(),
        "Release " << release.id() << " does not have issue_id");

    return Release{
        .issueId = (uint64_t) *release.issueId(),
        .releaseId = std::stoull(release.id())
    };
}

ReleaseGeometry
convertToReleaseGeometry(const sfactory::mosaics::Mosaic& mosaic)
{
    return ReleaseGeometry{
        .zmin = (int) mosaic.zoomMin(),
        .zmax = (int) mosaic.zoomMax(),
        .mercatorGeom =
            geolib3::convertGeodeticToMercator(
                geolib3::sproto::decode(*mosaic.geometry())
            )
    };
}

} // namespace

FactoryServiceClient::FactoryServiceClient()
{
    retryPolicy_ = maps::common::RetryPolicy()
        .setTryNumber(6)
        .setInitialCooldown(std::chrono::seconds(1))
        .setCooldownBackoff(2.);
}

http::Response
FactoryServiceClient::performRequest(http::Method method, const http::URL& url)
{
    return maps::common::retry(
        [&]() {
            maps::http::Request request(client_, method, url);

            maps::http::Response response = request.perform();
            if (!response.status_experimental().isServerError()) {
                return response;
            } else {
                throw maps::RuntimeError()
                    << "Failed perform request " << url.toString();
            }
        },
        maps::common::RetryPolicy()
            .setTryNumber(6)
            .setInitialCooldown(std::chrono::seconds(1))
            .setCooldownBackoff(2.)
    );
}

XMLFactoryServiceClient::XMLFactoryServiceClient(const std::string& url)
    : factoryUrl_(url)
{}

std::set<Release> XMLFactoryServiceClient::getAllReleases()
{
    static const std::string RELEASE_ID_ATTR = "id";
    static const std::string ISSUE_ID_ATTR = "issue-id";

    std::set<Release> releases;

    maps::http::URL url = factoryUrl_;
    url.setPath("/satrep/GetTreeNodes")
        .addParam("node_id", "release-status:production/releases");
    auto response = performRequest(http::GET, url);

    xml3::Doc releasesXML = xml3::Doc::fromString(response.readBody());
    xml3::Node root = releasesXML.root();
    xml3::Nodes nodes = root.nodes("node");

    for (size_t i = 0; i < nodes.size(); i++) {
        xml3::Node releaseNode = nodes[i].node("release");
        Release release;
        release.releaseId = releaseNode.attr<uint64_t>(RELEASE_ID_ATTR);
        release.issueId = releaseNode.attr<uint64_t>(ISSUE_ID_ATTR);
        releases.insert(release);
    }

    return releases;
}

ReleaseGeometries XMLFactoryServiceClient::loadReleaseGeometries(uint64_t releaseId)
{
    static const std::string SHAPES_ARCHIVE_NAME = "shapes.zip";

    TTempDir tmpDir;
    std::string archivePath = common::joinPath(tmpDir.Name(), SHAPES_ARCHIVE_NAME);

    maps::http::URL url = factoryUrl_;
    url.setPath("/boundary-exporter/")
        .addParam("id", "release:" + std::to_string(releaseId))
        .addParam("format", "shape");
    auto response = performRequest(http::GET, url);

    std::ofstream ofs(archivePath, std::ios::binary);
    REQUIRE(ofs.is_open(), "Failed to open file: " + archivePath);
    ofs << response.readBody();
    ofs.close();

    unzip(archivePath, tmpDir.Name());

    std::string shapePath = common::joinPath(
        tmpDir.Name(), "release_" + std::to_string(releaseId) + "-shape.shp");

    return readShapefile(shapePath);
}

ProtoFactoryServiceClient::ProtoFactoryServiceClient(const std::string& url)
    : factoryUrl_(url)
{}

std::set<Release> ProtoFactoryServiceClient::getAllReleases()
{
    std::set<Release> releases;

    maps::http::URL url = factoryUrl_;
    url.setPath("/v1/releases/search")
        .addParam("status", "production");
    auto response = performRequest(http::GET, url);
    const auto protoReleases = boost::lexical_cast<sfactory::release::Releases>(response.readBody());

    for (const auto& protoRelease : protoReleases.releases()) {
        releases.insert(convertToRelease(protoRelease));
    }

    return releases;
}

ReleaseGeometries ProtoFactoryServiceClient::loadReleaseGeometries(uint64_t releaseId)
{
    maps::http::URL url = factoryUrl_;
    url.setPath("/v1/mosaics/search")
        .addParam("release_id", std::to_string(releaseId));
    auto response = performRequest(http::GET, url);

    const auto protoMosaics = boost::lexical_cast<sfactory::mosaics::Mosaics>(response.readBody());
    ReleaseGeometries releaseGeometries;
    releaseGeometries.reserve(protoMosaics.mosaics().size());
    for (const auto& protoMosaic : protoMosaics.mosaics()) {
        releaseGeometries.push_back(convertToReleaseGeometry(protoMosaic));
    }
    return releaseGeometries;
}


} // namespace maps::wiki::autocart::pipeline
