#pragma once

#include <maps/libs/enum_io/include/enum_io_fwd.h>

#include <maps/libs/http/include/http.h>

#include <maps/libs/common/include/retry.h>

#include <memory>
#include <chrono>
#include <string>
#include <unordered_map>

namespace maps::wiki::autocart::pipeline::hitman {

const std::string SANDBOX_HITMAN_HOST = "sandbox.hitman.yandex-team.ru";
const std::string PRODUCTION_HITMAN_HOST = "hitman.yandex-team.ru";

using HitmanJobId = uint64_t;

struct HitmanJob {
    HitmanJobId id;
    std::string workflowId;
    std::string instanceId;
};

using HitmanProcessProperties = std::unordered_map<std::string, std::string>;

enum class HitmanJobStatus {
    NEW,
    RUNNING,
    START_FAILED,
    FAILED,
    CANCELED,
    SUCCEEDED,
    FIXED,
    RESUMING,
    IN_CANCELLATION
};

DECLARE_ENUM_IO(HitmanJobStatus);

struct ClientError : maps::Exception {
    using Exception::Exception;
};

struct ServerError : maps::Exception {
    using Exception::Exception;
};

struct ParallelRunsLimit : maps::Exception {
    using Exception::Exception;
};


class HitmanClient {
public:
    HitmanClient(const std::string& host, const std::string& authToken);


    const std::string& schema() const;
    HitmanClient& setSchema(std::string /*move*/ schema);

    std::chrono::milliseconds timeout() const;
    HitmanClient& setTimeout(std::chrono::milliseconds timeout);

    size_t maxRequestAttempts() const;
    HitmanClient& setMaxRequestAttempts(size_t maxRequestAttempts);

    std::chrono::milliseconds retryInitialTimeout() const;
    HitmanClient& setRetryInitialTimeout(std::chrono::milliseconds timeout);

    double retryTimeoutBackoff() const;
    HitmanClient& setRetryTimeoutBackoff(double retryTimeoutBackoff);


    HitmanJobId runProcess(
        const std::string& processCode,
        const std::string& requester,
        const HitmanProcessProperties& properties) const;

    HitmanJobId waitAvailableRunsAndRunProcess(
        const std::string& processCode,
        const std::string& requester,
        const HitmanProcessProperties& properties,
        const std::chrono::minutes& waitTimeout,
        const std::chrono::minutes& recheckInterval) const;

    HitmanJobStatus getJobStatus(const HitmanJobId& jobId) const;

    std::vector<HitmanJob> getJobsByStatus(
        const std::string& processCode, const HitmanJobStatus& status) const;

    HitmanProcessProperties getProcessProperties(const HitmanJobId& jobId) const;

private:
    std::string processParamsToJson(
        const std::string& requester, const HitmanProcessProperties& properties) const;

    http::Response performRequestChecked(http::Request& request) const;

    template <typename RequestFunctor>
    http::Response retry(RequestFunctor&& requestFunc) const;

    std::string host_;
    std::string authToken_;
    std::string schema_;

    std::chrono::milliseconds timeout_ = std::chrono::seconds(10);
    size_t maxRequestAttempts_ = 5;
    std::chrono::milliseconds retryInitialTimeout_ = std::chrono::seconds(1);
    double retryTimeoutBackoff_ = 2.;

    std::unique_ptr<http::Client> httpClient_;
};

} // namespace maps::wiki::autocart::pipeline::hitman
