#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/area.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/ft_type_id.h>

#include <maps/libs/common/include/exception.h>

#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/hex_wkb.h>

#include <mapreduce/yt/interface/client.h>

namespace maps::wiki::autocart::pipeline {

namespace {

constexpr const char* SHAPE = "shape";
constexpr const char* FT_TYPE_ID = "ft_type_id";
constexpr const char* AREA_LIST = "areas";

} // namespace

Area Area::fromYTNode(const NYT::TNode& node) {
    Area area;
    area.geoGeom_ = hexWKBToMultiPolygon(node[SHAPE].AsString());
    NYT::TNode ftTypeIdNode = node[FT_TYPE_ID];
    if (!node[FT_TYPE_ID].IsUndefined() && !node[FT_TYPE_ID].IsNull()) {
        area.setFTTypeId(decodeFTTypeId(node[FT_TYPE_ID].AsInt64()));
    }
    return area;
}

NYT::TNode Area::toYTNode() const {
    NYT::TNode node;
    toYTNode(node);
    return node;
}

void Area::toYTNode(NYT::TNode& node) const {
    node[SHAPE] = TString(multiPolygonToHexWKB(geoGeom_));
    if (hasFTTypeId()) {
        node[FT_TYPE_ID] = encodeFTTypeId(getFTTypeId());
    }
}

bool Area::hasFTTypeId() const {
    return ftTypeId_.has_value();
}

void Area::setFTTypeId(const FTTypeId& ftTypeId) {
    REQUIRE(isCompatibleWithArea(ftTypeId),
            "ft_type_id is not compatible with area: " << encodeFTTypeId(ftTypeId));
    ftTypeId_ = ftTypeId;
}

FTTypeId Area::getFTTypeId() const {
    REQUIRE(hasFTTypeId(), "ft_type_id is not defined");
    return ftTypeId_.value();
}

bool Area::operator==(const Area& that) const {
    return geolib3::test_tools::approximateEqual(geoGeom_, that.geoGeom_, geolib3::EPS)
        && ftTypeId_ == that.ftTypeId_;
}

std::vector<Area> areasFromYTNode(const NYT::TNode& node) {
    NYT::TNode::TListType areaList = node[AREA_LIST].AsList();
    size_t count = areaList.size();
    std::vector<Area> areas;
    areas.reserve(count);
    for (size_t i = 0; i < count; i++) {
        areas.push_back(Area::fromYTNode(areaList[i]));
    }
    return areas;
}

void areasToYTNode(const std::vector<Area>& areas, NYT::TNode& node) {
    NYT::TNode areaList = NYT::TNode::CreateList();
    for (const Area& area : areas) {
        areaList.Add(area.toYTNode());
    }
    node[AREA_LIST] = areaList;
}

} // namespace maps::wiki::autocart::pipeline
