#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/building.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/objects/include/ft_type_id.h>

#include <maps/libs/common/include/exception.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/hex_wkb.h>

#include <mapreduce/yt/interface/client.h>

namespace maps::wiki::autocart::pipeline {

namespace {

constexpr const char* SHAPE = "shape";
constexpr const char* HEIGHT = "height";
constexpr const char* FT_TYPE_ID = "ft_type_id";
constexpr const char* BLD_LIST = "blds";

void writePolygonToJson(const geolib3::Polygon2& polygon, json::ObjectBuilder& builder) {
    builder[SHAPE] = [&](json::ArrayBuilder b) {
        for (size_t i = 0; i < polygon.pointsNumber(); i++) {
            b << [&](json::ArrayBuilder b) {
                geolib3::Point2 point = polygon.pointAt(i);
                b << point.x() << point.y();
            };
        }
    };
}

} // namespace

Building Building::fromYTNode(const NYT::TNode& node) {
    Building bld;
    bld.geoGeom_ = hexWKBToPolygon(node[SHAPE].AsString());
    if (!node[BLD_ID].IsUndefined() && !node[BLD_ID].IsNull()) {
        bld.setId(node[BLD_ID].AsInt64());
    }
    if (!node[HEIGHT].IsUndefined() && !node[HEIGHT].IsNull()) {
        bld.setHeight(node[HEIGHT].AsInt64());
    }
    NYT::TNode ftTypeIdNode = node[FT_TYPE_ID];
    if (!node[FT_TYPE_ID].IsUndefined() && !node[FT_TYPE_ID].IsNull()) {
        bld.setFTTypeId(decodeFTTypeId(node[FT_TYPE_ID].AsInt64()));
    }
    return bld;
}

Building Building::fromJson(const json::Value& value) {
    Building bld;
    geolib3::PointsVector points;
    points.reserve(value[SHAPE].size());
    for (const json::Value& pointJson : value[SHAPE]) {
        points.emplace_back(pointJson[0].as<double>(), pointJson[1].as<double>());
    }
    bld.geoGeom_ = geolib3::Polygon2(points);
    if (value.hasField(BLD_ID) && !value[BLD_ID].isNull()) {
        bld.setId(value[BLD_ID].as<int64_t>());
    }
    if (value.hasField(HEIGHT) && !value[HEIGHT].isNull()) {
        bld.setHeight(value[HEIGHT].as<int>());
    }
    if (value.hasField(FT_TYPE_ID) && !value[FT_TYPE_ID].isNull()) {
        bld.setFTTypeId(decodeFTTypeId(value[FT_TYPE_ID].as<int>()));
    }
    return bld;
}

NYT::TNode Building::toYTNode() const {
    NYT::TNode node;
    toYTNode(node);
    return node;
}

void Building::toYTNode(NYT::TNode& node) const {
    node[SHAPE] = TString(polygonToHexWKB(geoGeom_));
    if (hasId()) {
        node[BLD_ID] = getId();
    }
    if (hasHeight()) {
        node[HEIGHT] = getHeight();
    }
    if (hasFTTypeId()) {
        node[FT_TYPE_ID] = encodeFTTypeId(getFTTypeId());
    }
}


void Building::toJson(json::ObjectBuilder& builder) const {
    writePolygonToJson(geoGeom_, builder);
    if (hasId()) {
        builder[BLD_ID] = getId();
    }
    if (hasHeight()) {
        builder[HEIGHT] = getHeight();
    }
    if (hasFTTypeId()) {
        builder[FT_TYPE_ID] = encodeFTTypeId(getFTTypeId());
    }

}

bool Building::hasId() const {
    return id_.has_value();
}

bool Building::hasHeight() const {
    return height_.has_value();
}

bool Building::hasFTTypeId() const {
    return ftTypeId_.has_value();
}

void Building::setId(int64_t id) {
    id_ = id;
}

void Building::setHeight(int height) {
    height_ = height;
}

void Building::setFTTypeId(const FTTypeId& ftTypeId) {
    REQUIRE(isCompatibleWithBld(ftTypeId),
            "ft_type_id is not compatible with bld: " << encodeFTTypeId(ftTypeId));
    ftTypeId_ = ftTypeId;
}

int64_t Building::getId() const {
    REQUIRE(hasId(), "bld_id is not defined");
    return id_.value();
}

int Building::getHeight() const {
    REQUIRE(hasHeight(), "height is not defined");
    return height_.value();
}

FTTypeId Building::getFTTypeId() const {
    REQUIRE(hasFTTypeId(), "ft_type_id is not defined");
    return ftTypeId_.value();
}

bool Building::operator==(const Building& that) const {
    return geolib3::test_tools::approximateEqual(geoGeom_, that.geoGeom_, geolib3::EPS)
        && height_ == that.height_
        && id_ == that.id_
        && ftTypeId_ == that.ftTypeId_;
}

std::vector<Building> bldsFromYTNode(const NYT::TNode& node) {
    NYT::TNode bldList = node[BLD_LIST];
    std::vector<Building> blds;
    blds.reserve(bldList.AsList().size());
    for (size_t i = 0;i < bldList.AsList().size(); i++) {
        blds.push_back(Building::fromYTNode(bldList[i]));
    }
    return blds;
}

void bldsToYTNode(const std::vector<Building>& blds, NYT::TNode& node) {
    NYT::TNode bldList = NYT::TNode::CreateList();
    for (const Building& bld : blds) {
        bldList.Add(bld.toYTNode());
    }
    node[BLD_LIST] = bldList;
}

} // namespace maps::wiki::autocart::pipeline
