#include <maps/libs/common/include/exception.h>

#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/yt_storage.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/strings.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/detection_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/tolokers_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/assessors_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/storage/include/publication_results.h>
#include <maps/wikimap/mapspro/services/autocart/pipeline/libs/yt_utils/include/rows_count.h>
#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/polygon_processing.h>

#include <maps/libs/log8/include/log8.h>

#include <maps/libs/chrono/include/time_point.h>

#include <mapreduce/yt/util/ypath_join.h>

#include <util/generic/size_literals.h>

#include <chrono>
#include <thread>

namespace maps::wiki::autocart::pipeline {

namespace {

static const TString ISSUE_ID_TABLE_NAME = "issue_id";
static const TString RELEASE_TABLE_NAME = "releases";
static const TString EXECUTION_TABLE_NAME = "execution";
static const TString COVERAGE_TABLE_NAME = "coverage";

static const TDuration WAIT_LOCK_DURATION = TDuration::Hours(12);

constexpr const char* DATE = "date";
constexpr const char* DATE_FORMAT = "%Y-%m-%d %H:%M:%S";

template <typename Result>
TString resultsTablePath(const TString& storagePath) {
    return NYT::JoinYPaths(storagePath, Result::getName());
}

} // namespace

void createYTStorage(NYT::ITransactionPtr txn, const TString& storagePath) {
    REQUIRE(!txn->Exists(storagePath),
            "Failed to create YT storage. Node " + storagePath + " already exists");
    txn->Create(
        storagePath, NYT::NT_MAP,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
    );
    txn->Create(
        NYT::JoinYPaths(storagePath, ISSUE_ID_TABLE_NAME), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
    );
    txn->Create(
        NYT::JoinYPaths(storagePath, RELEASE_TABLE_NAME), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
    );
    txn->Create(
        NYT::JoinYPaths(storagePath, COVERAGE_TABLE_NAME), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
    );
    txn->Create(
        NYT::JoinYPaths(storagePath, EXECUTION_TABLE_NAME), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
    );
    txn->Create(
        resultsTablePath<DetectionResult>(storagePath), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
            .Attributes(NYT::TNode()
                ("schema", DetectionResult::getTableSchema().ToNode()))
    );
    txn->Create(
        resultsTablePath<TolokersResult>(storagePath), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
            .Attributes(NYT::TNode()
                ("schema", TolokersResult::getTableSchema().ToNode()))
    );
    txn->Create(
        resultsTablePath<AssessorsResult>(storagePath), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
            .Attributes(NYT::TNode()
                ("schema", AssessorsResult::getTableSchema().ToNode()))
    );
    txn->Create(
        resultsTablePath<PublicationResult>(storagePath), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .IgnoreExisting(false)
            .Attributes(NYT::TNode()
                ("schema", PublicationResult::getTableSchema().ToNode()))
    );
}

YTStorageClient::YTStorageClient(
    NYT::IClientPtr client, const TString& storagePath)
    : client_(client),
      storagePath_(storagePath),
      issueIdTablePath_(NYT::JoinYPaths(storagePath, ISSUE_ID_TABLE_NAME)),
      releaseTablePath_(NYT::JoinYPaths(storagePath, RELEASE_TABLE_NAME)),
      coverageTablePath_(NYT::JoinYPaths(storagePath, COVERAGE_TABLE_NAME)),
      executionTablePath_(NYT::JoinYPaths(storagePath, EXECUTION_TABLE_NAME)),
      waitLockDuration_(WAIT_LOCK_DURATION)
{
    REQUIRE(
        client_->Exists(storagePath),
        "YT storage does not exist: " + storagePath
    );
    REQUIRE(
        client_->Exists(resultsTablePath<DetectionResult>()),
        "Table with detection result does not exist: "
        << resultsTablePath<DetectionResult>()
    );
    REQUIRE(
        client_->Exists(resultsTablePath<TolokersResult>()),
        "Table with tolokers result does not exist: "
        << resultsTablePath<TolokersResult>()
    );
    REQUIRE(
        client_->Exists(resultsTablePath<AssessorsResult>()),
        "Table with assessors result does not exist: "
        << resultsTablePath<AssessorsResult>()
    );
    REQUIRE(
        client_->Exists(resultsTablePath<PublicationResult>()),
        "Table with publication result does not exist: "
        << resultsTablePath<PublicationResult>()
    );
}

// AOI and issueId map methods

std::unordered_map<TString, ProcessedIssue>
YTStorageClient::getRegionToIssueMapWithoutLock(
    NYT::ITransactionPtr txn, const TString& lockedNodePath) const
{
    std::unordered_map<TString, ProcessedIssue> regionToIssue;
    NYT::TTableReaderPtr<NYT::TNode> issueReader
        = txn->CreateTableReader<NYT::TNode>(lockedNodePath);
    for (; issueReader->IsValid(); issueReader->Next()) {
        const NYT::TNode& row = issueReader->GetRow();
        regionToIssue[row[REGION].AsString()] = {
            row[ISSUE_ID].AsUint64(),
            chrono::parseIntegralDateTime(row[DATE].AsString(), DATE_FORMAT)
        };
    }
    return regionToIssue;
}

std::unordered_map<TString, ProcessedIssue>
YTStorageClient::getRegionToIssueMap() const {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        issueIdTablePath_,
        NYT::ELockMode::LM_SNAPSHOT,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);
    NYT::TNodeId nodeId = lock->GetLockedNodeId();
    TString nodePath = "#" + GetGuidAsString(nodeId);
    return getRegionToIssueMapWithoutLock(txn, nodePath);
}

std::optional<ProcessedIssue>
YTStorageClient::getIssue(const TString& region) const {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        issueIdTablePath_,
        NYT::ELockMode::LM_SNAPSHOT,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);
    NYT::TNodeId nodeId = lock->GetLockedNodeId();
    TString nodePath = "#" + GetGuidAsString(nodeId);
    std::unordered_map<TString, ProcessedIssue> regionToIssue
        = getRegionToIssueMapWithoutLock(txn, nodePath);
    auto it = regionToIssue.find(region);
    if (it != regionToIssue.end()) {
        return it->second;
    } else {
        return std::nullopt;
    }
}

void YTStorageClient::updateIssue(const TString& region, uint64_t issueId) {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        issueIdTablePath_,
        NYT::ELockMode::LM_EXCLUSIVE,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);
    std::unordered_map<TString, ProcessedIssue> regionToIssue
        = getRegionToIssueMapWithoutLock(txn, issueIdTablePath_);
    auto it = regionToIssue.find(region);
    if (it == regionToIssue.end()) {
        regionToIssue[region] = {issueId, chrono::TimePoint::clock::now()};
    } else if (it->second.issueId < issueId) {
        it->second = {issueId, chrono::TimePoint::clock::now()};;
    } else {
        INFO() << "Issue id for " << region << " has already been updated";
        return;
    }
    NYT::TTableWriterPtr<NYT::TNode> issueWriter
        = txn->CreateTableWriter<NYT::TNode>(issueIdTablePath_);
    for (const auto& [region, issue] : regionToIssue) {
        NYT::TNode node;
        node[REGION] = region;
        node[ISSUE_ID] = issue.issueId;
        node[DATE] = TString(chrono::formatIntegralDateTime(issue.date, DATE_FORMAT));
        issueWriter->AddRow(node);
    }
    issueWriter->Finish();
    txn->Commit();
}

void YTStorageClient::saveExecution(const BldRecognitionRegion& region, uint64_t issue_id) {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        executionTablePath_,
        NYT::ELockMode::LM_EXCLUSIVE,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    NYT::TNode node = region.toYTNode();
    node[ISSUE_ID] = issue_id;
    // UTC+0:00
    chrono::TimePoint date = chrono::TimePoint::clock::now();
    node[DATE] = TString(chrono::formatIntegralDateTime(date, DATE_FORMAT));

    NYT::TRichYPath appendYTPath(executionTablePath_);
    appendYTPath.Append(true);
    NYT::TTableWriterPtr<NYT::TNode> writer
        = txn->CreateTableWriter<NYT::TNode>(appendYTPath);
    writer->AddRow(node);
    writer->Finish();

    txn->Commit();
}

std::vector<Execution> YTStorageClient::getExecutionHistory() const {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        executionTablePath_,
        NYT::ELockMode::LM_SNAPSHOT,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    std::vector<Execution> history;

    NYT::TNodeId nodeId = lock->GetLockedNodeId();
    TString nodePath = "#" + GetGuidAsString(nodeId);
    NYT::TTableReaderPtr<NYT::TNode> reader
        = txn->CreateTableReader<NYT::TNode>(nodePath);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& node = reader->GetRow();
        Execution execution;
        execution.region = BldRecognitionRegion::fromYTNode(node);
        execution.issueId = node[ISSUE_ID].AsUint64();
        execution.date = chrono::parseIntegralDateTime(node[DATE].AsString(), DATE_FORMAT);
        history.push_back(execution);
    }

    return history;
}

// Satellite releases

bool YTStorageClient::hasReleases() const {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        releaseTablePath_,
        NYT::ELockMode::LM_SNAPSHOT,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    NYT::TNodeId nodeId = lock->GetLockedNodeId();
    TString nodePath = "#" + GetGuidAsString(nodeId);

    return getRowsCount(txn, nodePath) != 0;
}

Release YTStorageClient::getLastRelease() const {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        releaseTablePath_,
        NYT::ELockMode::LM_SNAPSHOT,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    NYT::TNodeId nodeId = lock->GetLockedNodeId();
    TString nodePath = "#" + GetGuidAsString(nodeId);

    size_t rowsCount = getRowsCount(txn, nodePath);
    REQUIRE(rowsCount > 0, "There is no releases in YT storage");

    NYT::TRichYPath lastRow(nodePath);
    lastRow.AddRange(NYT::TReadRange().FromRowIndices(rowsCount - 1, rowsCount));
    NYT::TTableReaderPtr<NYT::TNode> reader
        = txn->CreateTableReader<NYT::TNode>(lastRow);
    REQUIRE(reader->IsValid(), "Can not read last release geometry from YT storage");
    NYT::TNode row = reader->GetRow();

    return Release::fromYTNode(row);
}

void YTStorageClient::addRelease(
    const Release& release,
    const ReleaseAttrs& attrs)
{
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        releaseTablePath_,
        NYT::ELockMode::LM_EXCLUSIVE,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    bool isValidRelease = false;
    size_t rowsCount = getRowsCount(txn, releaseTablePath_);
    if (0u == rowsCount) {
        // can add any release in empty table
        isValidRelease = true;
    } else {
        NYT::TRichYPath lastRow(releaseTablePath_);
        lastRow.AddRange(NYT::TReadRange().FromRowIndices(rowsCount - 1, rowsCount));
        NYT::TTableReaderPtr<NYT::TNode> reader
            = txn->CreateTableReader<NYT::TNode>(lastRow);
        REQUIRE(reader->IsValid(), "Can not read last release geometry from YT storage");
        NYT::TNode row = reader->GetRow();
        Release lastRelease = Release::fromYTNode(row);
        if (lastRelease < release) {
            isValidRelease = true;
        }
    }
    REQUIRE(isValidRelease, "Can not add previuos release to YT storage");
    NYT::TRichYPath appendYPath(releaseTablePath_);
    appendYPath.Append(true);
    NYT::TTableWriterPtr<NYT::TNode> writer
        = txn->CreateTableWriter<NYT::TNode>(appendYPath);
    NYT::TNode releaseNode = release.toYTNode();
    for (const ReleaseGeometry& geometry : attrs.geometries) {
        geometry.toYTNode(releaseNode);
        releaseNode[DATE] = TString(chrono::formatIntegralDateTime(attrs.date, DATE_FORMAT));
        writer->AddRow(releaseNode);
    }
    writer->Finish();

    txn->Commit();
}

std::map<Release, ReleaseAttrs> YTStorageClient::getAllReleasesAttrs() const {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        releaseTablePath_,
        NYT::ELockMode::LM_SNAPSHOT,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    std::map<Release, ReleaseAttrs> releaseToAttrs;
    NYT::TNodeId nodeId = lock->GetLockedNodeId();
    TString nodePath = "#" + GetGuidAsString(nodeId);
    NYT::TTableReaderPtr<NYT::TNode> reader
        = txn->CreateTableReader<NYT::TNode>(nodePath);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& node = reader->GetRow();
        Release release = Release::fromYTNode(node);
        ReleaseGeometry geometry = ReleaseGeometry::fromYTNode(node);
        ReleaseAttrs& attrs = releaseToAttrs[release];
        attrs.geometries.push_back(geometry);
        attrs.date = chrono::parseIntegralDateTime(node[DATE].AsString(), DATE_FORMAT);
    }
    return releaseToAttrs;
}

// Releases coverage

void YTStorageClient::updateCoverage(const ReleasesCoverage& newCoverage) {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        coverageTablePath_,
        NYT::ELockMode::LM_EXCLUSIVE,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    std::vector<ReleasesCoverage> coverages;

    bool isExists = false;
    NYT::TTableReaderPtr<NYT::TNode> reader
        = txn->CreateTableReader<NYT::TNode>(coverageTablePath_);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& node = reader->GetRow();
        ReleasesCoverage coverage = ReleasesCoverage::fromYTNode(node);
        if (coverage.z != newCoverage.z) {
            coverages.push_back(coverage);
        } else {
            REQUIRE(coverage.lastRelease < newCoverage.lastRelease,
                    "There is coverage with later release in YT storage");
            coverages.push_back(newCoverage);
            isExists = true;
        }
    }
    if (!isExists) {
        coverages.push_back(newCoverage);
    }

    NYT::TTableWriterPtr<NYT::TNode> writer
        = txn->CreateTableWriter<NYT::TNode>(
            coverageTablePath_,
            NYT::TTableWriterOptions().Config(NYT::TNode()("max_row_weight", 128_MB))
        );
    for (const ReleasesCoverage& coverage : coverages) {
        writer->AddRow(coverage.toYTNode());
    }
    writer->Finish();

    txn->Commit();
}

geolib3::MultiPolygon2 YTStorageClient::updateCoverage(
    const std::map<Release, ReleaseAttrs>& releaseToAttrs,
    size_t zoom)
{
    if (releaseToAttrs.empty()) {
        return {};
    }
    Release lastRelease = releaseToAttrs.rbegin()->first;

    std::optional<ReleasesCoverage> coverage = getCoverageAtZoom(zoom);
    if (!coverage.has_value() || coverage->lastRelease < lastRelease) {
        ReleasesCoverage newCoverage;
        newCoverage.z = zoom;
        newCoverage.lastRelease = lastRelease;
        if (coverage.has_value()) {
            auto it = releaseToAttrs.upper_bound(coverage->lastRelease);
            newCoverage.mercatorGeom = cascadedMergeMultiPolygons({
                coverage->mercatorGeom,
                getReleasesCoverageAtZoom({it, releaseToAttrs.end()}, zoom)
            });
        } else {
            newCoverage.mercatorGeom = getReleasesCoverageAtZoom(releaseToAttrs, zoom);
        }
        updateCoverage(newCoverage);
        return newCoverage.mercatorGeom;
    } else {
        return coverage->mercatorGeom;
    }
}

std::optional<ReleasesCoverage> YTStorageClient::getCoverageAtZoom(int zoom) const {
    NYT::ITransactionPtr txn = client_->StartTransaction();
    NYT::ILockPtr lock = txn->Lock(
        coverageTablePath_,
        NYT::ELockMode::LM_SNAPSHOT,
        NYT::TLockOptions().Waitable(true)
    );
    lock->Wait(waitLockDuration_);

    NYT::TNodeId nodeId = lock->GetLockedNodeId();
    TString nodePath = "#" + GetGuidAsString(nodeId);
    NYT::TTableReaderPtr<NYT::TNode> reader
        = txn->CreateTableReader<NYT::TNode>(nodePath);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& node = reader->GetRow();
        ReleasesCoverage coverage = ReleasesCoverage::fromYTNode(node);
        if (coverage.z == zoom) {
            return coverage;
        }
    }

    return std::nullopt;
}

} // namespace maps::wiki::autocart::pipeline
