#include <maps/wikimap/mapspro/services/autocart/qa/metric/lib/model.h>
#include <maps/wikimap/mapspro/services/autocart/qa/metric/utm/utm.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/std.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/optional.hpp>
#include <boost/filesystem.hpp>
#include <geos/geom/Polygon.h>
#include <contrib/libs/gdal/ogr/ogrsf_frmts/ogrsf_frmts.h>
#include <opencv2/opencv.hpp>

#include <fstream>
#include <iterator>
#include <iomanip>
#include <cmath>
#include <memory>
#include <vector>


namespace fs = boost::filesystem;
namespace geolib3 = maps::geolib3;
namespace json = maps::json;

namespace {

struct MarkupTask {
    std::string task;
    std::string id;
    geolib3::Polygon2 polygon;
};


void readMarkupTasks(const std::string& filePath,
                     const std::string& taskName,
                     std::back_insert_iterator<std::vector<MarkupTask>> taskOuputIt)
{
    INFO() << "loading tasks from file " << filePath;
    auto tasksJson = json::Value::fromFile(filePath);

    REQUIRE(tasksJson.isArray(), "wrong file format: " << filePath);
    for(auto taskJson : tasksJson) {
        REQUIRE(taskJson.isObject(), "wrong file format: " << filePath);
        auto id = taskJson["id"].as<std::string>();
        auto coordsJson = taskJson["coords"];
        REQUIRE(coordsJson.isArray(), "wrong file format: " << filePath);
        std::vector<geolib3::Point2> points;
        points.reserve(coordsJson.size());
        for(auto coordJson : coordsJson) {
            points.emplace_back(coordJson[0].as<double>(), coordJson[1].as<double>());
        }
        try {
            *taskOuputIt = MarkupTask{taskName, id, geolib3::Polygon2(std::move(points))};
        } catch (const maps::Exception& ex) {
            ERROR() << "failed to read markup task " << ex;
        }

        ++taskOuputIt;
    }
}


std::vector<MarkupTask> readMarkupTasks(const std::string& dirPath)
{
    INFO() << "Loading markup tasks from dir " << dirPath;
    REQUIRE(fs::is_directory(dirPath), "not a directory");
    fs::path dir(dirPath);
    fs::directory_iterator endIt;
    std::vector<MarkupTask> results;
    for(fs::directory_iterator it(dir); it != endIt; ++it) {
        if(fs::is_regular_file(it->path())) {
            readMarkupTasks(it->path().string(), it->path().filename().string(), std::back_inserter(results));
        }
    }
    return results;
}

void
readPolygonsFromShapeFile(const std::string& filename,
                          std::back_insert_iterator<std::vector<geolib3::Polygon2>> polygonOuputIt)
{
    INFO() << "Loading polygons from " << filename;
    OGRRegisterAll();
    auto driver = GetGDALDriverManager()->GetDriverByName("ESRI Shapefile");
    ASSERT(driver);

    std::unique_ptr<GDALDataset> ds(driver->Create(filename.c_str(), 0, 0, 0, GDT_Unknown, nullptr));
    REQUIRE(ds, "Can't open file " << filename);
    ASSERT(ds->GetLayerCount() == 1);

    OGRLayer* layer = ds->GetLayer(0);
    for (int f = 0; f < layer->GetFeatureCount(); ++f) {
        OGRFeature* feature = layer->GetFeature(f);

        ASSERT(feature->GetGeometryRef()->getGeometryType() == wkbPolygon);
        OGRPolygon* geometry =
            static_cast<OGRPolygon*>(feature->GetGeometryRef());

        OGRLinearRing* exteriorRing = geometry->getExteriorRing();
        ASSERT(exteriorRing);

        std::vector<geolib3::Point2> points;
        points.reserve(exteriorRing->getNumPoints());

        for (int p = 0; p < exteriorRing->getNumPoints(); ++p) {
            points.emplace_back(exteriorRing->getX(p), exteriorRing->getY(p));
        }
        *polygonOuputIt = geolib3::Polygon2(std::move(points));
        polygonOuputIt++;
    }
}

std::vector<geolib3::Polygon2>
readPolygonsFromShapeDir(const std::string& dirPath)
{
    INFO() << "Loading polygons from dir " << dirPath;
    REQUIRE(fs::is_directory(dirPath), "not a directory");
    fs::path dir(dirPath);
    fs::directory_iterator endIt;
    std::vector<geolib3::Polygon2> results;
    for(fs::directory_iterator it(dir); it != endIt; ++it) {
        if(fs::is_regular_file(it->path()) && it->path().extension().string() == ".shp") {
            readPolygonsFromShapeFile(it->path().string(), std::back_inserter(results));
        }
    }
    return results;
}

std::vector<geolib3::Polygon2>
readPolygonsFromJson(const std::string& path)
{
    std::vector<geolib3::Polygon2> result;
    auto polygonsJson = json::Value::fromFile(path);
    ASSERT(polygonsJson.isArray());
    result.reserve(polygonsJson.size());
    for(auto polygonJson : polygonsJson) {
        ASSERT(polygonJson.isArray());
        std::vector<geolib3::Point2> points;
        points.reserve(polygonJson.size());
        for(auto coordJson : polygonJson) {
            points.emplace_back(coordJson[0].as<double>(),
                                coordJson[1].as<double>());
        }
        result.push_back(
            geolib3::Polygon2(
                geolib3::LinearRing2{points, false},
                {},
                false)
        );
    }
    return result;
}


class IoUMatcher {
public:
    using GeomHolder = std::unique_ptr<geos::geom::Geometry>;

    IoUMatcher(const std::vector<geolib3::Polygon2>& polygons)
    {
        for(const auto& polygon : polygons) {
            searcher_.insert(&polygon, 0);
        }
        searcher_.build();
    }

    boost::optional<std::pair<double, geolib3::Polygon2>>
    match(const geolib3::Polygon2& polygon) {
        constexpr double IOU_THRESHOLD = 0.1;
        boost::optional<std::pair<double, geolib3::Polygon2>> result;

        auto searchResult = searcher_.find(polygon.boundingBox());

        for(auto it = searchResult.first; it != searchResult.second; ++it){
            auto iou = calculateIoU(polygon, it->geometry());
            if (iou > IOU_THRESHOLD && (!result || result->first < iou)) {
                result = std::make_pair(iou, it->geometry());
            }
        }
        return result;
    }

    static double calculateIoU(const geolib3::Polygon2& polygon1,
                               const geolib3::Polygon2& polygon2)
    {
        auto geosPolygon1 = geolib3::internal::geolib2geosGeometry(polygon1);
        auto geosPolygon2 = geolib3::internal::geolib2geosGeometry(polygon2);

        GeomHolder intersection(geosPolygon1->intersection(geosPolygon2.get()));
        GeomHolder unionGeom(geosPolygon1->Union(geosPolygon2.get()));

        return intersection->getArea() / unionGeom->getArea();
    }

private:
    geolib3::StaticGeometrySearcher<geolib3::Polygon2, int> searcher_;
};


geolib3::Point2 convertCoordsToUTM(const geolib3::Point2& point) {
    long zone;
    char hemisphere;

    double lat = point.y();
    double lon = point.x();

    double outLat = 0.;
    double outLon = 0.;

    lat *= M_PI / 180.;
    lon *= M_PI / 180.;

    long error = Convert_Geodetic_To_UTM(lat, lon, &zone, &hemisphere, &outLat, &outLon);
    REQUIRE(error == UTM_NO_ERROR,
        "Failed to convert to UTM " << error  << outLon << " " << outLat);

    return {outLon, outLat};

}

maps::wiki::autocart::Polygon
toModelPolygon(const geolib3::Polygon2& polygon)
{
    auto exteriorRing = polygon.exteriorRing();
    std::vector<maps::wiki::autocart::Point> points;
    points.reserve(exteriorRing.pointsNumber() + 1);
    for(size_t i = 0; i < exteriorRing.pointsNumber(); ++i) {
        // auto point = geolib3::geoPoint2Mercator(exteriorRing.pointAt(i));
        // auto point = exteriorRing.pointAt(i);
        auto point = convertCoordsToUTM(exteriorRing.pointAt(i));
        points.emplace_back(point.x(), point.y());
        // std::cout << "(" << point.x() << ", " << point.y() << ") ";
    }
    points.emplace_back(points[0].x, points[0].y);
    // std::cout << std::endl;
    return points;
}

geolib3::Polygon2 convertToPolygon(const cv::RotatedRect& rect)
{
    std::vector<cv::Point2f> points(4);
    rect.points(points.data());
    std::vector<geolib3::Point2> ring;
    ring.reserve(4);
    for(const auto& point : points) {
        ring.emplace_back(point.x, point.y);
    }

    return geolib3::Polygon2(std::move(ring));
}

geolib3::Polygon2
toRectangle(const geolib3::Polygon2& polygon)
{
    auto exteriorRing = polygon.exteriorRing();
    std::vector<cv::Point2f> points;
    points.reserve(exteriorRing.pointsNumber());
    for(size_t i = 0; i < exteriorRing.pointsNumber(); ++i) {
        auto point = exteriorRing.pointAt(i);
        points.emplace_back(point.x(), point.y());
    }
    auto rect = cv::minAreaRect(points);
    return convertToPolygon(rect);
}

std::pair<double, maps::wiki::autocart::AffineResult>
predictMetric(const maps::wiki::autocart::Model& model,
                const maps::wiki::autocart::OptimizationParams& params,
                const geolib3::Polygon2& refPolygon,
                const geolib3::Polygon2& evalPolygon)
{
    auto modelRefPolygon = toModelPolygon(toRectangle(refPolygon));
    // auto modelRefPolygon = toModelPolygon(refPolygon);
    auto modelevalPolygon = toModelPolygon(evalPolygon);
    auto affine = maps::wiki::autocart::find_affine(
            modelRefPolygon, modelevalPolygon, params
        );
    auto res = model.predict(affine);
    return std::make_pair(res, affine);
}

void writePolygonsToFile(const std::vector<geolib3::Polygon2>& polygons,
                         const std::string& path)
{
    INFO() << "writing polygons to file " << path;
    std::ofstream f(path);
    REQUIRE(f.good(), "Can't open file " << path);
    f << std::setprecision(10);
    json::Builder builder(f);
    builder << [&](json::ArrayBuilder builder) {
        for(const auto& polygon : polygons) {
            auto exteriorRing = polygon.exteriorRing();
            builder << [&](json::ArrayBuilder builder) {
                for(size_t i = 0; i < exteriorRing.pointsNumber(); ++i) {
                    auto point = exteriorRing.pointAt(i);
                    builder << [&](json::ArrayBuilder builder) {
                        builder << point.x() << point.y();
                    };
                }
                auto point = exteriorRing.pointAt(0);
                builder << [&](json::ArrayBuilder builder) {
                    builder << point.x() << point.y();
                };
            };
        }
    };
}

} // namespace


int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;

    auto refPolygonsDirPath = parser.string("reference_polygons_dir")
        .help("path to shape file with reference polygons");

    auto markupTasksDirPath = parser.string("markup_tasks_dir")
        .help("dir with markup tasks");

    auto refPolygonsJsonPath = parser.string("reference_polygons_json");
    auto testPolygonsJsonPath = parser.string("test_polygons_json");

    auto wktBoundaryFilePath = parser.string("region_boundary")
        .help("file with wkt boundary of region");

    auto modelPath = parser.string("model_path")
        .required()
        .help("path to model file");


    auto outputTestPolygonsPath = parser.string("output_test_polygons");
    auto outputRefPolygonsPath = parser.string("output_ref_polygons");

    parser.parse(argc, argv);

    REQUIRE(refPolygonsDirPath.defined() && markupTasksDirPath.defined() ||
             refPolygonsJsonPath.defined() && testPolygonsJsonPath.defined(),
             "Both reference and test polygons must be defined");

    boost::optional<geolib3::Polygon2> regionBoundary;

    REQUIRE(!modelPath.empty(), "model not defined");
    INFO() << "Loading model " << modelPath;
    maps::wiki::autocart::LinearModel model(modelPath.c_str());

    if(wktBoundaryFilePath.defined()) {
        auto boundaryWkt = maps::common::readFileToString(wktBoundaryFilePath);
        regionBoundary = geolib3::WKT::read<geolib3::Polygon2>(boundaryWkt);
    }

    std::vector<geolib3::Polygon2> matchedTestPolygons;
    std::vector<geolib3::Polygon2> matchedRefPolygons;

    // auto refPolygons = readPolygonsFromShapeFile(refPolygonsFilePath);
    std::vector<geolib3::Polygon2> refPolygons;
    if (refPolygonsDirPath.defined()) {
        refPolygons = readPolygonsFromShapeDir(refPolygonsDirPath);
    } else {
        refPolygons = readPolygonsFromJson(refPolygonsJsonPath);
    }

    INFO() << "Loaded " << refPolygons.size() << " reference polygons";

    maps::wiki::autocart::OptimizationParamsBuilder builder;
    auto optimizationParams = builder.build();

    const std::string SEP = "\t";
    std::cout << std::setprecision(9);

    if (markupTasksDirPath.defined()) {
        auto markupTasks = readMarkupTasks(markupTasksDirPath);
        INFO() << "Loaded " << markupTasks.size() << " evaluation polygons";


        INFO() << "Building matcher";
        IoUMatcher matcher(refPolygons);
        INFO() << "Done";

        int matched = 0;

        INFO() << "Matching polygons";

        int cnt = 0;
        for(const auto& task : markupTasks) {
            auto maybeMatched = matcher.match(task.polygon);

            if(!maybeMatched) {
                continue;
            }
            ++matched;

            auto metric = predictMetric(model, optimizationParams,
                    maybeMatched->second, task.polygon);

            int pointsDiff = task.polygon.exteriorRing().pointsNumber()
                - maybeMatched->second.exteriorRing().pointsNumber();

            std::cout
                << task.task << SEP
                << task.id << SEP
                << maybeMatched->first << SEP // IoU
                << metric.first << SEP // confidence
                << metric.second.residual << SEP
                << metric.second.transform.shift_x << SEP
                << metric.second.transform.shift_y << SEP
                << metric.second.transform.theta << SEP
                << metric.second.transform.scale << SEP
                << pointsDiff << SEP;

            const auto& polygon = maybeMatched->second;
            auto ring = polygon.exteriorRing();
            for(size_t i = 0; i < ring.pointsNumber(); ++i) {
                auto point = ring.pointAt(i);
                std::cout << point.x() << SEP << point.y() << SEP;
            }
            std::cout << std::endl;

            matchedTestPolygons.push_back(task.polygon);
            matchedRefPolygons.push_back(maybeMatched->second);
            ++cnt;
            // if (cnt > 10)
            //     break;
        }

        INFO() << "Matched " << matched << " polygons, "
            << markupTasks.size() - matched << " unmatched";
    } else {
        auto testPolygons = readPolygonsFromJson(testPolygonsJsonPath);
        INFO() << "Loaded " << testPolygons.size() << " test polygons";
        ASSERT(refPolygons.size() == testPolygons.size());
        for (size_t i = 0; i < refPolygons.size() && i; ++i) {
            auto metric = predictMetric(model, optimizationParams,
                                        refPolygons[i], testPolygons[i]);

            std::cout
                << metric.first << SEP // confidence
                << metric.second.residual << SEP
                << metric.second.transform.shift_x << SEP
                << metric.second.transform.shift_y << SEP
                << metric.second.transform.theta << SEP
                << metric.second.transform.scale << SEP
                << std::endl;
        }
    }

    if (outputTestPolygonsPath.defined()) {
        writePolygonsToFile(matchedTestPolygons, outputTestPolygonsPath);
    }

    if (outputRefPolygonsPath.defined()) {
        writePolygonsToFile(matchedRefPolygons, outputRefPolygonsPath);
    }


    return EXIT_SUCCESS;

}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
