#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/const.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/stringutils/include/join.h>
#include <maps/libs/tile/include/const.h>
#include <maps/libs/tile/include/utils.h>
#include <maps/infra/yacare/include/parse.h>

#include <maps/wikimap/mapspro/services/autocart/libs/satellite/include/load_sat_image.h>
#include <maps/wikimap/mapspro/libs/tf_inferencer/tf_inferencer.h>
#include <maps/wikimap/mapspro/services/autocart/libs/detection/include/detect_bld.h>
#include <maps/wikimap/mapspro/services/autocart/libs/post_processing/include/align_angles.h>
#include <maps/wikimap/mapspro/services/autocart/libs/post_processing/include/post_processing.h>

#include <contrib/libs/gdal/ogr/ogrsf_frmts/ogrsf_frmts.h>
#include <opencv2/opencv.hpp>
#include <boost/filesystem.hpp>

#include <functional>
#include <memory>
#include <vector>

namespace ac = maps::wiki::autocart;
namespace geolib3 = maps::geolib3;
namespace tf_inferencer = maps::wiki::tf_inferencer;

namespace {

const int SAT_IMAGE_ZOOM = 18;

template<class Converter>
geolib3::BoundingBox convert(const geolib3::BoundingBox& bbox, Converter converter)
{
    return geolib3::BoundingBox(converter(bbox.lowerCorner()),
                                converter(bbox.upperCorner()));
}

std::vector<geolib3::Polygon2>
rectifyAndAlignPolygons(const std::vector<geolib3::Polygon2>& polygons)
{
    std::vector<geolib3::Polygon2> boundingBoxes;
    boundingBoxes.reserve(polygons.size());

    for (const auto& polygon : polygons)
    try {
        boundingBoxes.push_back(ac::getBoundingRectangle(polygon));
    } catch (const std::exception& ex) {
        WARN() << "error till processing polygon "
            << geolib3::WKT::toString(polygon)
            << " :" << ex.what();
    }

    std::vector<geolib3::Polygon2> result =
        ac::alignRectangles(boundingBoxes, 3, 10, 200, 5);
    return result;
}

using BldRecognitionFunc =
    std::function<
        std::vector<ac::Polygon>
            (const tf_inferencer::TensorFlowInferencer&,
             const tf_inferencer::TensorFlowInferencer&,
             const cv::Mat&)>;


const std::unordered_map<std::string, BldRecognitionFunc>
    RECOGNITION_MODE_TO_FUNC  = {
        {   std::string("rectangle"), ac::detectBldByMinAreaRect },
        {
            std::string("edge"),
            [](const tf_inferencer::TensorFlowInferencer& /*segmentator*/,
               const tf_inferencer::TensorFlowInferencer& edgeDetector,
               const cv::Mat& image)
            {
                return ac::detectBldByEdges(edgeDetector, image);
            }
        }
    };

std::vector<geolib3::Polygon2>
convertToMercatorPolygons(const std::vector<ac::Polygon> &polygons,
                          const geolib3::Point2 &origin,
                          size_t zoom)
{
    std::vector<geolib3::Polygon2> result;
    result.reserve(polygons.size());

    for (const auto& polygon : polygons) {
        std::vector<geolib3::Point2> ring;
        ring.reserve(polygon.size());
        for(const auto& point : polygon) {
            ring.push_back(
                ac::imageToMercator(
                    geolib3::Point2(point.x, point.y), origin, zoom)
            );
        }
        result.emplace_back(std::move(ring));
    }
    return result;
}

geolib3::Polygon2
convertMercatorToGeoPolygon(const geolib3::Polygon2& polygon)
{
    return geolib3::convertMercatorToGeodetic(polygon);
}

std::vector<geolib3::Polygon2>
convertMercatorToGeoPolygons(const std::vector<geolib3::Polygon2>& polygons)
{
    std::vector<geolib3::Polygon2> result;
    result.reserve(polygons.size());
    std::transform(polygons.begin(), polygons.end(),
                   std::back_inserter(result), convertMercatorToGeoPolygon);
    return result;
}


class SpatialReferenceHolder {
public:
    SpatialReferenceHolder(int epsgCode)
    {
        sRef_.importFromEPSG(epsgCode);
    }

    OGRSpatialReference& get()
    {
        return sRef_;
    }
private:
    OGRSpatialReference sRef_;
};

void dumpToShapeFile(const int startId,
                     const std::vector<geolib3::Polygon2>& polygons,
                     const std::string& dirPath,
                     const std::string& name)
{
    OGRRegisterAll();
    auto driver =  GetGDALDriverManager()->GetDriverByName("ESRI Shapefile");

    std::unique_ptr<GDALDataset> dataSource(driver->Create(dirPath.c_str(), 0, 0, 0, GDT_Unknown, nullptr));
    REQUIRE(dataSource, "Can't open output data source");

    constexpr int EPSG_ID_WGS84 = 4326;
    static SpatialReferenceHolder sRef(EPSG_ID_WGS84);

    OGRLayer* layer = dataSource->CreateLayer(name.c_str(), &sRef.get(),
                                              OGRwkbGeometryType::wkbPolygon);
    REQUIRE(layer != 0, "Failed to create layer");
    OGRFieldDefn ogrField("id", OGRFieldType::OFTInteger);
    REQUIRE(layer->CreateField(&ogrField) == OGRERR_NONE,
            "Can't create field");


    int polygonId = startId;
    auto* layerDefn = layer->GetLayerDefn();
    for(const auto& polygon : polygons) {
        auto* feature = OGRFeature::CreateFeature(layerDefn);
        maps::concurrent::ScopedGuard fGuard([&]{ OGRFeature::DestroyFeature(feature); });
        feature->SetField("id", ++polygonId);

        OGRLinearRing ring;
        auto exteriorRing = polygon.exteriorRing();
        for (size_t idx=0; idx<exteriorRing.pointsNumber(); ++idx) {
            auto point = exteriorRing.pointAt(idx);
            ring.addPoint(point.x(), point.y());
        }
        OGRPolygon geom;
        geom.addRing(&ring);
        feature->SetGeometry(&geom);
        REQUIRE(layer->CreateFeature(feature) == OGRERR_NONE,
           "Failed to create feature ");
    }
}

} // namespace


int main(int argc, char** argv)
try {

    std::vector<std::string> modes;
    std::transform(RECOGNITION_MODE_TO_FUNC.begin(), RECOGNITION_MODE_TO_FUNC.end(),
                   std::back_inserter(modes),
                   [](decltype(RECOGNITION_MODE_TO_FUNC)::value_type kv) {
                       return kv.first;
                   });

    const auto supportedModes = maps::stringutils::join(modes, ", ");


    maps::cmdline::Parser parser;


    auto bboxStr = parser.string("bbox")\
        .required()\
        .help("region bbox in format lt_x,lt_y~rb_x,rb_y in geodetic coordinates");

    auto mode = parser.string("mode")\
        .required()\
        .help("recognition algorithm, one of: " + supportedModes);

    auto semSegPath = parser.string("semseg_path")\
        .required()\
        .help("path to the gdef file for semantic segmentation net");

    auto edgeDetectPath = parser.string("edge_detect_path")\
        .required()\
        .help("path to the gdef file for edge detector net");


    auto outputDir = parser.string("out_dir")\
        .required()
        .help("directory for output layer");

    auto outputName = parser.string("out_layer")\
        .required()\
        .help("output file name");

    auto startId = parser.num("start_id")\
        .defaultValue(1)\
        .help("Initial id for output object");

    parser.parse(argc, argv);

    if (!RECOGNITION_MODE_TO_FUNC.count(mode)) {
        std::cerr << "unsupported mode " << mode;
        return EXIT_FAILURE;
    }

    INFO() << "loading semseg model " << semSegPath;
    maps::wiki::tf_inferencer::TensorFlowInferencer semSeg(semSegPath);

    INFO() << "loading edge detection model " << edgeDetectPath;
    maps::wiki::tf_inferencer::TensorFlowInferencer edgeDetect(edgeDetectPath);

    auto geoBbox = yacare::Parser<geolib3::BoundingBox>()(bboxStr);
    auto mercatorBbox = convert(geoBbox, &geolib3::geoPoint2Mercator);

    INFO() << "Loading sat image";
    auto image = ac::loadSatImage(mercatorBbox, SAT_IMAGE_ZOOM);

    INFO() << "Detecting buildings";
    auto rasterPolygons =
        RECOGNITION_MODE_TO_FUNC.at(mode)(semSeg, edgeDetect, image);
    INFO() << "Detected " << rasterPolygons.size() << " buildings";

    geolib3::Point2 origin = ac::getDisplayOrigin(mercatorBbox, SAT_IMAGE_ZOOM);

    std::vector<geolib3::Polygon2> mercatorPolygons =
        convertToMercatorPolygons(rasterPolygons, origin, SAT_IMAGE_ZOOM);

    INFO() << "Rectifying and aligning buildings";
    auto alignedMercatorPolygons = rectifyAndAlignPolygons(mercatorPolygons);

    INFO() << "Converting to geo polygons";
    auto geoPolygons = convertMercatorToGeoPolygons(alignedMercatorPolygons);

    dumpToShapeFile(startId, geoPolygons, outputDir, outputName);

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
