import cv2
import numpy as np
from collections import namedtuple

Rect = namedtuple('Rect', ['x', 'y', 'width', 'height', 'angle'])

def load(filename):
    """Loads rectangles from text file.

    Args:
        filename (str): Path to text file with rectangles. Each line must
            contain 5 real numders in the following order: x, y, width, height, angle.

    Returns:
        list: List of named tuples Rect with rectangles parameters.
            Rect = namedtuple('Rect', ['x', 'y', 'width', 'height', 'angle'])

    Raises:
        IOError: If failed to open file.
        ValueError: If file contains a string in the wrong format.
    """
    try:
        rects_file = open(filename, 'r')
    except IOError:
        raise IOError("Failed to open file with rectangles: {}".format(filename))
    rects = []
    for line in rects_file:
        tokens = line.strip().split()
        try:
            x, y, w, h, angle = [float(item) for item in tokens]
        except:
            raise ValueError("Failed to convert line to rectangle: {}".format(line))

        rects.append(Rect(x, y, w, h, angle))
    rects_file.close()
    return rects

def save(filename, rects):
    """Save rectangles to text file.

    Args:
        filename (str): Path to text file with rectangles.
        rects (list): List of named tuples Rect with rectangles parameters.
            Rect = namedtuple('Rect', ['x', 'y', 'width', 'height', 'angle'])

    Raises:
        IOError: If failed to create file.
    """
    try:
        rects_file = open(filename, 'w')
    except IOError:
        raise IOError("Failed to create file with rectangles: {}".format(filename))
    for rect in rects:
        rects_file.write("{} {} {} {} {}\n".format(rect.x, rect.y, rect.width, rect.height, rect.angle))
    rects_file.close()


def draw(canvas, rects):
    """Draw list of rectangles.

    Args:
        canvas (np.array): Image with one or three channels.
        rects (list): List of named tuples Rect with rectangles parameters.
            Rect = namedtuple('Rect', ['x', 'y', 'width', 'height', 'angle'])

    Raise:
        ValueError: If got invalid image or rectangle.
    """
    if isinstance(canvas, np.ndarray):
        if canvas.ndim != 2 and canvas.ndim != 3:
            raise ValueError("Canvas must have 2 or 3 dimensions.")
        if canvas.ndim == 3:
            if canvas.shape[2] != 1 and canvas.shape[2] != 3:
                raise ValueError("Canvas must have 1 or 3 channels.")
    else:
        raise ValueError("Canvas must be numpy array.")
    if canvas.ndim == 3:
        color = [0, 255, 0]
    else:
        color = 255
    for rect in rects:
        pts = list(cv2.boxPoints(((rect.x, rect.y), (rect.width, rect.height), rect.angle)))
        pts.append(pts[0])
        for i in range(len(pts)-1):
            pt0 = (pts[i][0], pts[i][1])
            pt1 = (pts[i+1][0], pts[i+1][1])
            cv2.line(canvas, pt0, pt1, color, 3)
