#pragma once
#include <set>
#include <map>
#include <vector>
#include <iterator>
#include <algorithm>

namespace maps {
namespace wiki {
namespace autocart {

typedef std::vector<int>  VerticesPath;
typedef std::vector<int>  EdgesPath;

template <typename VertexData, typename EdgeData>
class GraphImpl {
public:
    template <typename T>
    struct VertexImpl {
        int m_index;
        std::set<int> m_edges;
        T m_data;
    };

    template <typename T>
    struct EdgeImpl {
        int m_index;
        int m_vertex1, m_vertex2;
        int getOverVertex(int vertex) const
        {
            if (vertex == m_vertex1)
                return m_vertex2;
            else if (vertex == m_vertex2)
                return m_vertex1;
            return -1;
        }
        T m_data;
    };

    using Vertex        = VertexImpl<VertexData>;
    using Edge          = EdgeImpl<EdgeData>;
    using GraphIndices  = GraphImpl<int, int>;

    using VerticesMap   = std::map<int, Vertex>;
    using EdgesMap      = std::map<int, Edge>;
public:
    GraphImpl()
        : m_lastVertexIdx(-1)
        , m_lastEdgeIdx(-1)
    {

    }
    ~GraphImpl()
    {

    }

    void clear()
    {
        m_lastVertexIdx = -1;
        m_lastEdgeIdx = -1;

        m_vertices.clear();
        m_edges.clear();
    }

    ////////////////////////////////////
    // methods for work with vertices //
    ////////////////////////////////////
    int addVertex(const VertexData &data)
    {
        m_lastVertexIdx++;
        Vertex vertex;
        vertex.m_index = m_lastVertexIdx;
        vertex.m_data = data;
        m_vertices.emplace(m_lastVertexIdx, vertex);
        return m_lastVertexIdx;
    }
    void eraseVertex(int vertex)
    {
        auto it = m_vertices.find(vertex);
        if (it == m_vertices.end())
            throw std::runtime_error(cv::format("Unable to erase vertex with index: %d", vertex));

        for (auto edge : it->second.m_edges)
        {
            auto ite = m_edges.find(edge);
            if (ite == m_edges.end())
                continue;

            const int other_vert = ite->second.getOverVertex(vertex);
            auto itv = m_vertices.find(other_vert);
            if (itv != m_vertices.end())
                itv->second.m_edges.erase(edge);
            m_edges.erase(ite);
        }
        m_vertices.erase(it);
    }

    auto beginVertices()
    {
        return m_vertices.begin();
    }
    auto beginVertices() const
    {
        return m_vertices.begin();
    }
    auto endVertices()
    {
        return m_vertices.end();
    }
    auto endVertices() const
    {
        return m_vertices.end();
    }

    auto findVertex(int vertex) const
    {
        return m_vertices.find(vertex);
    }
    size_t verticesSize() const
    {
        return m_vertices.size();
    }

    /////////////////////////////////
    // methods for work with edges //
    /////////////////////////////////
    int addEdge(int vertex1, int vertex2, EdgeData data)
    {
        auto it1 = m_vertices.find(vertex1);
        auto it2 = m_vertices.find(vertex2);
        if (it1 == m_vertices.end() || it2 == m_vertices.end())
            throw std::runtime_error(cv::format("Unable to create edge because invalid vertices index: %d, %d", vertex1, vertex2));

        m_lastEdgeIdx++;
        Edge edge;
        edge.m_index = m_lastEdgeIdx;
        edge.m_vertex1 = vertex1;
        edge.m_vertex2 = vertex2;
        edge.m_data = data;
        it1->second.m_edges.insert(m_lastEdgeIdx);
        it2->second.m_edges.insert(m_lastEdgeIdx);
        m_edges.emplace(m_lastEdgeIdx, edge);
        return m_lastEdgeIdx;
    }
    void eraseEdge(int edge)
    {
        auto it = m_edges.find(edge);
        if (it == m_edges.end())
            throw std::runtime_error(cv::format("Unable to erase edge with index: %d", edge));

        eraseEdge(it);
    }
    void eraseEdge(int vertex1, int vertex2)
    {
        //TODO OPT
        int edge = findCommonEdge(vertex1, vertex2);
        if (-1 == edge)
            return;

        auto it = m_edges.find(edge);
        if (it == m_edges.end())
            throw std::runtime_error(cv::format("Unable to erase edge with index: %d", edge));

        eraseEdgeInVertex(edge, it->second.m_vertex1);
        eraseEdgeInVertex(edge, it->second.m_vertex2);

        m_edges.erase(it);
    }
    auto eraseEdge(typename EdgesMap::iterator it)
    {
        eraseEdgeInVertex(it->first, it->second.m_vertex1);
        eraseEdgeInVertex(it->first, it->second.m_vertex2);

        return m_edges.erase(it);
    }

    auto beginEdges()
    {
        return m_edges.begin();
    }
    auto beginEdges() const
    {
        return m_edges.begin();
    }
    auto endEdges()
    {
        return m_edges.end();
    }
    auto endEdges() const
    {
        return m_edges.end();
    }

    auto findEdge(int edge) const
    {
        return m_edges.find(edge);
    }
    size_t edgesSize() const
    {
        return m_edges.size();
    }
public:
    /////////////////////////////////
    // complex methods             //
    /////////////////////////////////
    void eraseLeaves()
    {
        bool doit = true;
        while (doit)
        {
            doit = false;
            for (auto it : m_vertices)
            {
                if (it.second.m_edges.size() <= 1)
                {
                    eraseVertex(it.first);
                    doit = true;
                    break;
                }
            }
        }
    }
    //forest[i] - graph with vertices data equal ids of vertices this Graph, edges data equal ids of edges this Graph
    void extractSpanningForest(std::vector< GraphIndices > &forest) const
    {
        std::map<int, int> verts;
        for (auto it = m_vertices.begin(); it != m_vertices.end(); it++)
        {
            verts[it->first] = -1;
        }

        int rest_verts = (int)m_vertices.size();

        while (0 != rest_verts)
        {
            std::vector<int> last_verts;
            for (auto it = verts.begin(); it != verts.end(); it++)
            {
                if (-1 == it->second)
                {
                    last_verts.push_back(it->first);
                    break;
                }
            }

            GraphIndices graph;
            while (!last_verts.empty())
            {
                int vert_id = last_verts.back(); last_verts.pop_back();
                const Vertex &vert = m_vertices.find(vert_id)->second;

                if (-1 == verts[vert_id])
                {
                    verts[vert_id] = graph.addVertex(vert_id);
                    rest_verts--;
                }

                for (auto it = vert.m_edges.begin(); it != vert.m_edges.end(); it++)
                {
                    const Edge &edge = m_edges.find(*it)->second;
                    int vert_over = edge.getOverVertex(vert_id);
                    if (-1 != verts[vert_over])
                        continue;
                    int temp = graph.addVertex(vert_over);
                    verts[vert_over] = temp;
                    rest_verts--;

                    graph.addEdge(verts[vert_id], temp, *it);
                    last_verts.push_back(vert_over);
                }
            }
            if (0 < graph.edgesSize())
                forest.emplace_back(graph);
        }
    }
    //cycles[i] - contain fundamental cycles related to i-th connected component of graph
    void extractFundamentalCyclesVert(std::vector< std::vector< VerticesPath  >>& cycles) const
    {
        std::vector< GraphIndices > forest;
        extractSpanningForest(forest);

        cycles.resize(forest.size());

        std::set<int> edges_mark;
        std::map<int, std::pair<int, int> > verts_map; //index of vertex (in this graph) map to index of graph in the forest and index of vertrex in this graph
        for (int i = 0;  i < (int)forest.size(); i++)
        {
            const auto &graph = forest[i];
            for (auto it = graph.beginEdges(); it != graph.endEdges(); ++it)
            {
                edges_mark.insert(it->second.m_data);
            }
            for (auto it = graph.beginVertices(); it != graph.endVertices(); ++it)
            {
                verts_map[it->second.m_data] = std::pair<int, int>(i, it->first);
            }
        }

        for (auto it = m_edges.begin(); it != m_edges.end(); ++it)
        {
            if (edges_mark.find(it->first) != edges_mark.end())
                continue;
            int forest_idx = -1;

            const auto itv1 = verts_map.find(it->second.m_vertex1);
            if (itv1 == verts_map.end())
                throw std::runtime_error("Unable to forest of the vertex");
            const auto itv2 = verts_map.find(it->second.m_vertex2);
            if (itv2 == verts_map.end())
                throw std::runtime_error("Unable to forest of the vertex");

            if (itv1->second.first != itv2->second.first)
                throw std::runtime_error("The edge connects vertices from different graph of spanning forest");

            VerticesPath path;
            auto &graph = forest[itv1->second.first];
            graph.findPathVerts(itv1->second.second, itv2->second.second, path);
            for (int i = 0; i < (int)path.size(); i++)
            {
                path[i] = graph.findVertex(path[i])->second.m_data;
            }
            cycles.emplace_back(path);
        }
    }
    void extractFundamentalCyclesEdges(std::vector<std::vector< EdgesPath > >& cycles) const
    {
        std::vector< GraphIndices > forest;
        extractSpanningForest(forest);
        std::cout << "forest count: " << forest.size() << std::endl;
        cycles.resize(forest.size());

        std::set<int> edges_mark;
        std::map<int, std::pair<int, int> > verts_map; //index of vertex (in this graph) map to index of graph in the forest and index of vertrex in this graph
        for (int i = 0; i < (int)forest.size(); i++)
        {
            const auto &graph = forest[i];
            for (auto it = graph.beginEdges(); it != graph.endEdges(); ++it)
            {
                edges_mark.insert(it->second.m_data);
            }
            for (auto it = graph.beginVertices(); it != graph.endVertices(); ++it)
            {
                verts_map[it->second.m_data] = std::pair<int, int>(i, it->first);
            }
        }

        for (auto it = m_edges.begin(); it != m_edges.end(); ++it)
        {
            if (edges_mark.find(it->first) != edges_mark.end())
                continue;
            int forest_idx = -1;

            const auto itv1 = verts_map.find(it->second.m_vertex1);
            if (itv1 == verts_map.end())
                throw std::runtime_error("Unable to forest of the vertex");
            const auto itv2 = verts_map.find(it->second.m_vertex2);
            if (itv2 == verts_map.end())
                throw std::runtime_error("Unable to forest of the vertex");

            if (itv1->second.first != itv2->second.first)
                throw std::runtime_error("The edge connects vertices from different graph of spanning forest");

            forest_idx = itv1->second.first;
            EdgesPath path;
            auto &graph = forest[itv1->second.first];
            graph.findPathEdges(itv1->second.second, itv2->second.second, path);
            for (int i = 0; i < (int)path.size(); i++)
            {
                path[i] = graph.findEdge(path[i])->second.m_data;
            }
            path.push_back(it->first);
            cycles[forest_idx].emplace_back(path);
        }
    }
    void findPathVerts(int vertex1, int vertex2, VerticesPath &path) const
    {
        std::map<int, int> prev_vert;
        prev_vert[vertex1] = -1;

        std::vector<int> last_verts(1, vertex1);
        while (!last_verts.empty())
        {
            int vert_id = last_verts.back(); last_verts.pop_back();
            const Vertex &vert = m_vertices.find(vert_id)->second;
            for (auto it = vert.m_edges.begin(); it != vert.m_edges.end(); it++)
            {
                const Edge &edge = m_edges.find(*it)->second;
                int vert_over = edge.getOverVertex(vert_id);
                if (vert_over == vertex2)
                {
                    path.push_back(vertex2);
                    while (vert_id != vertex1)
                    {
                        path.push_back(vert_id);
                        vert_id = prev_vert[vert_id];
                    }
                    path.push_back(vertex1);
                    return;
                }
                if (prev_vert.find(vert_over) == prev_vert.end())
                {
                    prev_vert[vert_over] = vert_id;
                    last_verts.push_back(vert_over);
                }
            }
        }
    }
    void findPathEdges(int vertex1, int vertex2, EdgesPath &path) const
    {
        //std::pair<int, int> - first is id of prev vertex, second is id of edge to prev vert
        std::map<int, std::pair<int, int>> prev_vert;
        prev_vert[vertex1] = std::pair<int, int>(-1, -1);

        std::vector<int> last_verts(1, vertex1);
        while (!last_verts.empty())
        {
            int vert_id = last_verts.back(); last_verts.pop_back();
            const Vertex &vert = m_vertices.find(vert_id)->second;
            for (auto it = vert.m_edges.begin(); it != vert.m_edges.end(); it++)
            {
                const Edge &edge = m_edges.find(*it)->second;
                int vert_over = edge.getOverVertex(vert_id);
                if (vert_over == vertex2)
                {
                    path.push_back(edge.m_index);
                    while (vert_id != vertex1)
                    {
                        path.push_back(prev_vert[vert_id].second);
                        vert_id = prev_vert[vert_id].first;
                    }
                    return;
                }
                if (prev_vert.find(vert_over) == prev_vert.end())
                {
                    prev_vert[vert_over] = std::pair<int, int>(vert_id, edge.m_index);
                    last_verts.push_back(vert_over);
                }
            }
        }
    }


    //  woEdge != -1    ,      
    bool isVertsConnected(int vertex1, int vertex2, int woEdge = -1) const
    {
        std::map<int, int> prev_vert;
        prev_vert[vertex1] = -1;

        std::vector<int> last_verts(1, vertex1);
        while (!last_verts.empty())
        {
            int vert_id = last_verts.back(); last_verts.pop_back();
            const Vertex &vert = m_vertices.find(vert_id)->second;
            for (auto it = vert.m_edges.begin(); it != vert.m_edges.end(); it++)
            {
                if (woEdge == (*it))
                    continue;
                const Edge &edge = m_edges.find(*it)->second;
                int vert_over = edge.getOverVertex(vert_id);
                if (vert_over == vertex2)
                    return true;
                if (prev_vert.find(vert_over) == prev_vert.end())
                {
                    prev_vert[vert_over] = vert_id;
                    last_verts.push_back(vert_over);
                }
            }
        }
        return false;
    }
    static bool hasCommonEdge(const Vertex &v1, const Vertex &v2)
    {
        std::set<int> intersect;
        std::set_intersection(v1.m_edges.begin(), v1.m_edges.end(), v2.m_edges.begin(), v2.m_edges.end(), std::inserter(intersect, intersect.begin()));
        return (!intersect.empty());
    }
private:
    int m_lastVertexIdx;
    int m_lastEdgeIdx;

    VerticesMap m_vertices;
    EdgesMap m_edges;

    void eraseEdgeInVertex(int edge, int vertex)
    {
        auto it = m_vertices.find(vertex);
        if (it == m_vertices.end())
            throw std::runtime_error(cv::format("Unable to erase edge with index: %d in vertex with index: %d", edge, vertex));
        it->second.m_edges.erase(edge);
    }
    int findCommonEdge(int vertex1, int vertex2) const
    {
        auto it1 = m_vertices.find(vertex1);
        auto it2 = m_vertices.find(vertex2);
        if (it1 == m_vertices.end() || it2 == m_vertices.end())
            throw std::runtime_error(cv::format("Unable to find edge because invalid vertices index: %d, %d", vertex1, vertex2));

        std::set<int> intersect;
        std::set_intersection(it1->second.m_edges.begin(), it1->second.m_edges.end(), it2->second.m_edges.begin(), it2->second.m_edges.end(), std::inserter(intersect, intersect.begin()));

        if (intersect.empty())
            return -1;
        return *(intersect.begin());
    }
};

} //namespace autocart
} //namespace wiki
} //namespace maps
