#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <maps/libs/common/include/file_utils.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/hex_wkb.h>
#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/polygon_processing.h>
#include <maps/wikimap/mapspro/services/autocart/libs/satellite/include/load_sat_image.h>

#include <maps/libs/tile/include/utils.h>

#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/builder.h>

#include <util/generic/size_literals.h>

#include <fstream>

using namespace maps;
using namespace maps::common;
using namespace maps::geolib3;
using namespace maps::wiki::autocart;

namespace {

static const char* ITEMS = "items";
static const char* IMAGE = "image";
static const char* FT_ID = "ft_id";
static const char* SHAPE = "shape";
static const char* FT_TYPE_ID = "ft_type_id";

static const size_t LOGGING_STEP = 100;

struct Item {
    uint64_t ftId;
    uint64_t ftTypeId;
    MultiPolygon2 geom;
};

using ItemSearcher = StaticGeometrySearcher<MultiPolygon2, const Item&>;

std::vector<Item> loadDataset(const std::string& path) {
    std::vector<Item> dataset;
    for (const json::Value& itemJson : json::Value::fromFile(path)) {
        Item item;
        item.ftId = itemJson[FT_ID].as<uint64_t>();
        item.ftTypeId = itemJson[FT_TYPE_ID].as<uint64_t>();
        item.geom = convertGeodeticToMercator(
            hexWKBToMultiPolygon(itemJson[SHAPE].as<std::string>())
        );
        dataset.push_back(item);
    }
    return dataset;
}

class MercatorToImage {
public:
    MercatorToImage(const BoundingBox& bbox, size_t zoom)
        : zoom_(zoom)
    {
        tile::DisplayCoord lb = tile::mercatorToDisplay(bbox.lowerCorner(), zoom);
        tile::DisplayCoord tr = tile::mercatorToDisplay(bbox.upperCorner(), zoom);
        origin_ =  Point2(lb.x(), tr.y());
    }

    Point2 mercatorToImage(const Point2& geom) const {
        tile::DisplayCoord dispPt = tile::mercatorToDisplay(geom, zoom_);
        return Point2(dispPt.x() - origin_.x(), dispPt.y() - origin_.y());
    }

    LinearRing2 mercatorToImage(const LinearRing2& geom) const {
        std::vector<Point2> points;
        for (size_t i = 0; i < geom.pointsNumber(); i++) {
            points.push_back(mercatorToImage(geom.pointAt(i)));
        }
        return LinearRing2(points);
    }

    Polygon2 mercatorToImage(const Polygon2& geom) const {
        LinearRing2 exteriorRing = mercatorToImage(geom.exteriorRing());
        std::vector<LinearRing2> interiorRings;
        for (size_t i = 0; i < geom.interiorRingsNumber(); i++) {
            interiorRings.push_back(mercatorToImage(geom.interiorRingAt(i)));
        }
        return Polygon2(exteriorRing, interiorRings);
    }

    MultiPolygon2 mercatorToImage(const MultiPolygon2& geom) const {
        std::vector<Polygon2> polygons;
        for (size_t i = 0; i < geom.polygonsNumber(); i++) {
            polygons.push_back(mercatorToImage(geom.polygonAt(i)));
        }
        return MultiPolygon2(polygons);
    }

private:
    size_t zoom_;
    Point2 origin_;
};

BoundingBox getImageBBox(const cv::Mat& image) {
    return BoundingBox(
        {0., 0.},
        {static_cast<double>(image.cols), static_cast<double>(image.rows)}
    );
};

BoundingBox getBoundingBox(const MultiPolygon2 geom, double padRatio, double minSize) {
    BoundingBox bbox = resizeByRatio(geom.boundingBox(), padRatio);
    if (bbox.width() < minSize || bbox.height() < minSize) {
        double width = std::max(minSize, bbox.width());
        double height = std::max(minSize, bbox.height());
        bbox = BoundingBox(bbox.center(), width, height);
    }
    return bbox;
}

std::vector<Item> selectItemsInBBox(
    const BoundingBox& bbox,
    const ItemSearcher& searcher)
{
    std::vector<Item> items;
    auto searchResult = searcher.find(bbox);
    for (auto it = searchResult.first; it != searchResult.second; it++) {
        items.push_back(it->value());
    }
    return items;
}

std::vector<Item> convertToImageItems(
    const std::vector<Item> mercItems,
    const BoundingBox& bbox,
    size_t zoom)
{
    MercatorToImage converter(bbox, zoom);
    std::vector<Item> imageItems;
    for (Item item : mercItems) {
        item.geom = converter.mercatorToImage(item.geom);
        imageItems.push_back(item);
    }
    return imageItems;
}

void intersectWithBBox(std::vector<Item>& items, const BoundingBox& bbox) {
    MultiPolygon2 mp({bbox.polygon()});
    for (Item& item : items) {
        item.geom = intersectMultiPolygons(item.geom, mp);
    }
}

void dumpToJson(
    const std::string& imageName,
    const std::vector<Item>& items,
    json::ObjectBuilder& b)
{
    b[IMAGE] = imageName;
    b[ITEMS] = [&](json::ArrayBuilder b) {
        for (const Item& item : items) {
            b << [&](json::ObjectBuilder b) {
                b[FT_ID] = item.ftId;
                b[FT_TYPE_ID] = item.ftTypeId;
                b[SHAPE] = multiPolygonToHexWKB(item.geom);
            };
        }
    };
}

} // namespace

int main(int argc, const char** argv)
try {
    maps::cmdline::Parser parser("Load images for ft_type_id dataset");

    maps::cmdline::Option<std::string> inputJsonPath = parser.string("input")
        .required()
        .help("Path to input json file with dataset");

    maps::cmdline::Option<std::string> imagesFolder = parser.string("images")
        .required()
        .help("Path to output folder with images");

    maps::cmdline::Option<std::string> outputJsonPath = parser.string("output")
        .required()
        .help("Path to output json file with dataset");

    maps::cmdline::Option<double> padRatio = parser.real("pad")
        .required()
        .help("Pad ratio");

    maps::cmdline::Option<double> mercSize = parser.real("min_merc_size")
        .required()
        .help("Minimal size of bounding box in mercator");

    maps::cmdline::Option<size_t> zoom = parser.size_t("zoom")
        .required()
        .help("Satellite image zoom");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Loading dataset: " << inputJsonPath;
    std::vector<Item> dataset = loadDataset(inputJsonPath);
    INFO() << "Dataset size: " << dataset.size();

    INFO() << "Building geometry searcher";
    ItemSearcher searcher;
    for (size_t i = 0; i < dataset.size(); i++) {
        searcher.insert(&(dataset[i].geom), dataset[i]);
    }
    searcher.build();
    INFO() << "Builded";

    INFO() << "Loading images and saving results";
    size_t count = 0;
    size_t totalCount = dataset.size();
    std::ofstream ofs(outputJsonPath);
    REQUIRE(ofs.is_open(), "Failed to create file: " + outputJsonPath);
    json::Builder builder(ofs);
    builder << [&](json::ArrayBuilder b) {
        for (const Item& item : dataset) {
            count++;
            if (count % LOGGING_STEP == 0) {
                INFO() << "Processing item " << count << "/" << totalCount;
            }
            BoundingBox bbox = getBoundingBox(item.geom, padRatio, mercSize);
            cv::Mat image;
            try {
                image = loadSatImage(bbox, zoom);
            } catch (const std::exception& e) {
                WARN() << "Failed to load image: " << e.what();
                continue;
            }

            std::vector<Item> mercItems = selectItemsInBBox(bbox, searcher);

            std::vector<Item> imageItems;
            try {
                imageItems = convertToImageItems(mercItems, bbox, zoom);
            } catch (const std::exception& e) {
                WARN() << "Failed to convert to image coordinates: " << e.what();
                continue;
            }

            intersectWithBBox(imageItems, getImageBBox(image));
            imageItems.erase(
                std::remove_if(
                    imageItems.begin(), imageItems.end(),
                    [](const Item& item) {
                        return item.geom.polygonsNumber() == 0;
                    }
                ),
                imageItems.end()
            );

            std::string imageName = std::to_string(item.ftId) + ".jpg";
            std::string imagePath = joinPath(imagesFolder, imageName);
            cv::imwrite(imagePath, image);
            b << [&](json::ObjectBuilder b) {
                dumpToJson(imageName, imageItems, b);
            };
        }
    };
    ofs.close();

    INFO() << "Done!";

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
