#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/shell_cmd.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/bounding_box.h>

#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/hex_wkb.h>
#include <maps/wikimap/mapspro/services/autocart/libs/satellite/include/load_sat_image.h>
#include <maps/wikimap/mapspro/services/autocart/libs/utils/include/multithreading.h>

#include <maps/libs/tile/include/utils.h>

#include <boost/algorithm/string.hpp>
#include <opencv2/opencv.hpp>

#include <optional>

namespace maps::wiki::autocart {

using Building = geolib3::Polygon2;

struct Item {
    Building bld;
    std::string state;
};

struct ItemWithImage {
    Building bld;
    std::string state;
    cv::Mat image;
    cv::Mat mask;
};

class ItemReader : public autocart::DataReader<Item> {
public:
    ItemReader(const std::string& inputListPath)
        : inputListPath_(inputListPath) {}

private:
    void Read() override {
        constexpr size_t PREFETCH_SIZE = 500;
        std::ifstream ifs(inputListPath_);
        while (!ifs.eof()) {
            std::string line;
            std::getline(ifs, line);
            if (line.empty()) {
                continue;
            }
            std::vector<std::string> items;
            boost::split(items, line, boost::is_any_of(","));
            Building bld = autocart::hexWKBToPolygon(items[0]);
            outQueue_->Enqueue({bld, items[1]});
            for (;;) {
                auto counter = outQueue_->GetCounter();
                if (PREFETCH_SIZE > static_cast<size_t>(AtomicGet(counter.Counter))) {
                    break;
                }
                std::this_thread::sleep_for(std::chrono::milliseconds(20));
            }
        }
        ifs.close();
    }

    std::string inputListPath_;
};

class ImageDownloader : public autocart::DataProcessor<Item, ItemWithImage> {
public:
    ImageDownloader(size_t zoom, double gapSize)
        : zoom_(zoom),
          gapSize_(gapSize) {}

    ImageDownloader* Clone() const override {
        return new ImageDownloader(*this);
    }

private:
    std::optional<ItemWithImage> Do(const Item& item) override {
        ItemWithImage itemWithImage;
        itemWithImage.bld = item.bld;
        itemWithImage.state = item.state;

        Building mercatorBld = geolib3::convertGeodeticToMercator(item.bld);
        geolib3::BoundingBox bbox = mercatorBld.boundingBox();
        bbox = geolib3::resizeByRatio(bbox, 1 + gapSize_);

        cv::Mat image;
        try {
            image = autocart::loadSatImage(bbox, zoom_);
        } catch (...) {
            return std::nullopt;
        }
        itemWithImage.image = image;

        cv::Mat mask;
        drawBuilding(bbox, mercatorBld, mask);
        itemWithImage.mask = mask;

        return itemWithImage;
    }

    void drawBuilding(const geolib3::BoundingBox& bbox,
                      const Building& mercatorBld,
                      cv::Mat& mask) {
        auto firstCorner = tile::mercatorToDisplaySigned(bbox.lowerCorner(), zoom_);
        auto secondCorner = tile::mercatorToDisplaySigned(bbox.upperCorner(), zoom_);
        tile::SignedDisplayCoord leftTop(std::min(firstCorner.x(), secondCorner.x()),
                                       std::min(firstCorner.y(), secondCorner.y()));
        size_t maskWidth = abs(firstCorner.x() - secondCorner.x()) + 1;
        size_t maskHeight = abs(firstCorner.y() - secondCorner.y()) + 1;
        mask = cv::Mat(maskHeight, maskWidth, CV_8UC1, cv::Scalar::all(0));

        std::vector<cv::Point2i> polygon;
        for (size_t i = 0; i < mercatorBld.pointsNumber(); i++) {
            geolib3::Point2 point = mercatorBld.pointAt(i);
            tile::SignedDisplayCoord dispPoint = tile::mercatorToDisplaySigned(point, zoom_);
            polygon.emplace_back(dispPoint.x() - leftTop.x(), dispPoint.y() - leftTop.y());
        }

        std::vector<std::vector<cv::Point2i>> polygons(1, polygon);
        cv::fillPoly(mask, polygons, cv::Scalar::all(255));
    }

    size_t zoom_;
    double gapSize_;
};

void createDirectory(const std::string& path) {
    shell::Result result = shell::runCmd("mkdir -p " + path);
    REQUIRE(result.exitCode == 0, result.stdErr);
}

class ItemWriter : public autocart::DataWriter<ItemWithImage> {
public:
    ItemWriter(const std::string& datasetPath)
        : datasetPath_(datasetPath),
          imagePath_(common::joinPath(datasetPath_, "image")),
          maskPath_(common::joinPath(datasetPath_, "mask")),
          index_(0) {
        createDirectory(imagePath_);
        createDirectory(maskPath_);
        dataListOfs_.open(common::joinPath(datasetPath_, "list.txt"));
    }

private:
    void Write(const ItemWithImage& item) override {
        if (index_ % 100 == 0) {
            INFO() << "Processed " << index_ << " items";
        }
        dataListOfs_ << index_ << "," << item.state << "\n";
        cv::imwrite(common::joinPath(imagePath_, std::to_string(index_) + ".jpg"), item.image);
        cv::imwrite(common::joinPath(maskPath_, std::to_string(index_) + ".png"), item.mask);
        index_++;
    }

    virtual void Finish() {
        dataListOfs_.close();
    }

    std::string datasetPath_;
    std::string imagePath_;
    std::string maskPath_;
    std::ofstream dataListOfs_;
    size_t index_;
};

void downloadImagesAndMasks(const std::string& inputListPath,
                            const std::string& datasetPath,
                            size_t zoom,
                            double gapSize,
                            size_t threadsNum) {
    MultithreadDataProcessor<Item, ItemWithImage> mtp;
    mtp.SetReader(new ItemReader(inputListPath));
    mtp.SetProcessors(new ImageDownloader(zoom, gapSize), threadsNum);
    mtp.SetWriter(new ItemWriter(datasetPath));
    mtp.Run();
}

} // namespace maps::wiki::autocart

int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser("Downloading dataset for AutoToloker");

    auto inputListPath = parser.string("input_list")
        .required()
        .help("Path to text file with coordinates and toloka answers");

    auto datasetPath = parser.string("dataset_path")
        .required()
        .help("Path to dataset");

    auto zoom = parser.size_t("zoom")
        .required()
        .help("Satellite image zoom");

    auto gapSize = parser.real("gap_size")
        .required()
        .help("Size of gap (in percents)");

    auto threadsNum = parser.size_t("threads_num")
        .required()
        .help("Threads number");

    parser.parse(argc, argv);

    INFO() << "Downloading images and masks";
    maps::wiki::autocart::downloadImagesAndMasks(
        inputListPath,
        datasetPath,
        zoom,
        gapSize,
        threadsNum
    );

    INFO() << "Done!";

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
