#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <pqxx/pqxx>
#include <boost/optional.hpp>

#include <string>
#include <vector>
#include <regex>

namespace common = maps::wiki::common;
namespace geolib3 = maps::geolib3;
namespace rev = maps::wiki::revision;

namespace {

using Date = std::string;

bool isValidDate(const Date& date) {
    return std::regex_match(date, std::regex("[0-9]{4}-[0-9]{2}-[0-9]{2}"));
}

struct BldRecognitionTask {
    BldRecognitionTask(const size_t& taskId,
                       const geolib3::Polygon2& recognitionArea,
                       const Date& createdAt,
                       const std::string& mode)
        : id(taskId)
        , aoi(recognitionArea)
        , date(createdAt)
        , mode(mode)
    {}

    size_t id;
    geolib3::Polygon2 aoi;
    Date date;
    std::string mode;
};

struct BldRecognitionStats {
    // количество запущенных заданий
    size_t tasksCnt = 0;
    // количество верно найденных зданий
    size_t truePositiveCnt = 0;
    // количество ложно найденных зданий
    size_t falsePositiveCnt = 0;
    // количество необнаруженных зданий
    size_t falseNegativeCnt = 0;

    double getPrecision() const {
        if (truePositiveCnt + falsePositiveCnt == 0) {
            return 1.;
        } else {
            return truePositiveCnt / double(truePositiveCnt + falsePositiveCnt);
        }
    }

    double getRecall() const {
        if (truePositiveCnt + falseNegativeCnt == 0) {
            return 1.;
        } else {
            return truePositiveCnt / double(truePositiveCnt + falseNegativeCnt);
        }
    }

    BldRecognitionStats& operator+=(const BldRecognitionStats& stats) {
        this->tasksCnt += stats.tasksCnt;
        this->truePositiveCnt += stats.truePositiveCnt;
        this->falsePositiveCnt += stats.falsePositiveCnt;
        this->falseNegativeCnt += stats.falseNegativeCnt;
        return *this;
    }
};

std::ostream& operator<<(std::ostream& os, const BldRecognitionStats& stats) {
    os << "Precision: " << stats.getPrecision() << "\n";
    os << "Recall: " << stats.getRecall() << "\n";
    os << "Number of tasks: " << stats.tasksCnt << "\n";
    os << "Correctly found buildings: " << stats.truePositiveCnt << "\n";
    os << "Incorrectly found buildings: " << stats.falsePositiveCnt << "\n";
    os << "Not found buildings: " << stats.falseNegativeCnt << "\n";
    return os;
}

std::vector<BldRecognitionTask>
loadBldRecognitionTasks(pqxx::transaction_base& txn, const Date& minDate) {
    std::vector<BldRecognitionTask> tasks;

    std::stringstream sstream;
    sstream << "SELECT id, ST_AsBinary(aoi) aoi, created, mode\n"
            << "FROM service.bld_recognition_task\n"
            << "JOIN service.task USING(id)\n"
            << "ORDER BY id";
    auto query = sstream.str();

    for (const auto& row : txn.exec(query)) {
        size_t taskId = row["id"].as<size_t>();
        std::string geomWkb = pqxx::binarystring(row["aoi"]).str();
        std::stringstream geomWkbStream(geomWkb);
        geolib3::Polygon2 aoi = geolib3::WKB::read<geolib3::Polygon2>(geomWkbStream);
        std::string mode = row["mode"].as<std::string>();

        Date date = row["created"].as<std::string>();
        if (date >= minDate) {
            INFO() << date;
            tasks.emplace_back(taskId, aoi, date, mode);
        }
    }

    return tasks;
}

std::vector<geolib3::Polygon2>
loadBldRecognitionResult(pqxx::transaction_base &txn,
                         const BldRecognitionTask& task) {
    std::vector<geolib3::Polygon2> blds;

    std::stringstream sstream;
    sstream << "SELECT ST_AsBinary(shape) bld\n"
            << "FROM service.bld_recognition_result\n"
            << "WHERE task_id=" << task.id;
    auto query = sstream.str();

    for (const auto& row : txn.exec(query)) {
        std::string geomWkb = pqxx::binarystring(row["bld"]).str();
        std::stringstream geomWkbStream(geomWkb);
        geolib3::Polygon2 bld = geolib3::WKB::read<geolib3::Polygon2>(geomWkbStream);
        blds.push_back(std::move(bld));
    }

    return blds;
}

size_t getRevisionCommitIdBeforeTask(pqxx::transaction_base& txn,
                                     const BldRecognitionTask& task) {
    std::stringstream sstream;
    sstream << "SELECT commit_id\n"
            << "FROM social.commit_event\n"
            << "WHERE created_at < '" << task.date << "'\n"
            << "ORDER BY created_at DESC\n"
            << "LIMIT 1";
    auto query = sstream.str();
    auto result = txn.exec(query);
    size_t commitId = result[0]["commit_id"].as<size_t>();
    return commitId;
}

std::vector<geolib3::Polygon2>
loadBldsInPolygon(const rev::Snapshot& snapshot,
                  const geolib3::Polygon2& aoi) {
    std::vector<geolib3::Polygon2> blds;

    geolib3::BoundingBox bbox = aoi.boundingBox();
    auto bldsInBBox = snapshot.objectRevisionsByFilter(
            rev::filters::Attr("cat:bld").defined() &&
            rev::filters::Geom::intersects(bbox.minX(), bbox.minY(), bbox.maxX(), bbox.maxY())
        );

    for (const auto& bld : bldsInBBox) {
        if (bld.data().geometry) {
            std::stringstream ss(*bld.data().geometry);
            geolib3::Polygon2 polygon = geolib3::WKB::read<geolib3::Polygon2>(ss);
            if (geolib3::spatialRelation(polygon, aoi,
                                         geolib3::SpatialRelation::Within)) {
                blds.push_back(polygon);
            }
        }
    }

    return blds;
}

std::vector<geolib3::Polygon2>
loadBldsExistedBefore(pqxx::transaction_base& geomReadTxn,
                      pqxx::transaction_base& socialReadTxn,
                      const BldRecognitionTask& task){
    INFO() << "Getting revision commit id";
    size_t commitId = getRevisionCommitIdBeforeTask(socialReadTxn, task);
    INFO() << "Commit id: " << commitId;
    rev::RevisionsGateway gtw(geomReadTxn);
    auto snapshot = gtw.snapshot(commitId);

    return loadBldsInPolygon(snapshot, task.aoi);
}

std::vector<geolib3::Polygon2>
loadExistsBlds(pqxx::transaction_base& txn,
              const BldRecognitionTask& task) {
    rev::RevisionsGateway gtw(txn);
    auto snapshot = gtw.snapshot(gtw.headCommitId());

    return loadBldsInPolygon(snapshot, task.aoi);
}

bool isEqualPolygons(const geolib3::Polygon2& poly1,
                     const geolib3::Polygon2& poly2) {
    constexpr double DIST_EPS = 1e-3;
    if (poly1.pointsNumber() != poly2.pointsNumber()) {
        return false;
    }

    size_t ptsCnt = poly1.pointsNumber();

    for (size_t shift = 0; shift < ptsCnt; shift++) {
        bool isEqual = true;
        for (size_t i = 0; i < ptsCnt; i++) {
            geolib3::Point2 pt1 = poly1.pointAt(i);
            geolib3::Point2 pt2 = poly2.pointAt((i + shift) % ptsCnt);
            if (geolib3::distance(pt1, pt2) > DIST_EPS) {
                isEqual = false;
                break;
            }
        }
        if (isEqual) {
            return true;
        }
    }

    return false;
}

boost::optional<BldRecognitionStats>
getTaskStats(pqxx::transaction_base& geomReadTxn,
             pqxx::transaction_base& socialReadTxn,
             const BldRecognitionTask& task) {
    INFO() << "Task: " << task.id;
    BldRecognitionStats stats;
    INFO() << "Loading recognition result";
    std::vector<geolib3::Polygon2>
            detectedBlds = loadBldRecognitionResult(geomReadTxn, task);
    INFO() << "Loaded buildings: " << detectedBlds.size();

    INFO() << "Loading exists building";
    std::vector<geolib3::Polygon2>
            existsBlds = loadExistsBlds(geomReadTxn, task);
    INFO() << "Loaded buildings: " << existsBlds.size();

    geolib3::StaticGeometrySearcher<geolib3::Polygon2, size_t> searcher;
    for (size_t i = 0; i < existsBlds.size(); i++) {
        searcher.insert(&existsBlds[i], i);
    }
    searcher.build();

    size_t equalBldsCnt = 0;
    for (const auto& detectedBld : detectedBlds) {
        auto searchResult = searcher.find(detectedBld.boundingBox());
        for (auto it = searchResult.first; it != searchResult.second; it++) {
            size_t bldIndx = it->value();
            if (isEqualPolygons(existsBlds[bldIndx], detectedBld)) {
                equalBldsCnt++;
                break;
            }
        }
    }

    if (equalBldsCnt == 0) {
        // Считаем тестовым запуском
        INFO() << "Task " << task.id << " is test execution";
        return boost::none;
    }

    INFO() << "Loading exists before building";
    std::vector<geolib3::Polygon2>
            bldsExistedBefore = loadBldsExistedBefore(geomReadTxn, socialReadTxn, task);
    INFO() << "Loaded buildings: " << bldsExistedBefore.size();
    stats.tasksCnt = 1;
    stats.truePositiveCnt  = equalBldsCnt;
    stats.falsePositiveCnt = detectedBlds.size() - equalBldsCnt;
    stats.falseNegativeCnt = existsBlds.size() - bldsExistedBefore.size() - equalBldsCnt;

    return stats;
}

using ModeToStats = std::unordered_map<std::string, BldRecognitionStats>;

ModeToStats
getBldRecognitionStats(const common::ExtendedXmlDoc& configDoc, const Date& minDate) {
    common::PoolHolder coreDbHolder(configDoc, "core", "core");
    common::PoolHolder socialDbHolder(configDoc, "social", "grinder");
    auto geomReadTxn = coreDbHolder.pool().slaveTransaction();
    auto socialReadTxn = socialDbHolder.pool().slaveTransaction();
    std::vector<BldRecognitionTask> tasks = loadBldRecognitionTasks(*geomReadTxn, minDate);
    INFO() << "Tasks number: " << tasks.size();

    ModeToStats modeToStats;
    for (const auto& task : tasks) {
        auto taskStats = getTaskStats(*geomReadTxn, *socialReadTxn, task);
        if (taskStats) {
            modeToStats[task.mode] += *taskStats;
        }
    }

    return modeToStats;
}

} // namespace

int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;
    auto config = parser.string("config")\
        .help("path to wiki services.xml config file");
    auto minDate = parser.string("min_date")\
        .defaultValue("0000-00-00")\
        .help("Minimum task date");
    parser.parse(argc, argv);

    REQUIRE(isValidDate(minDate), "Date is invalid");

    std::unique_ptr<maps::wiki::common::ExtendedXmlDoc> configDocPtr;
    if (config.defined()) {
        configDocPtr.reset(new maps::wiki::common::ExtendedXmlDoc(config));
    } else {
        configDocPtr = maps::wiki::common::loadDefaultConfig();
    }

    INFO() << "Extract information from database";
    ModeToStats modeToStats = getBldRecognitionStats(*configDocPtr, minDate);
    BldRecognitionStats totalStats;

    for (const auto& [mode, stats] : modeToStats) {
        INFO() << "Recognition mode: " << mode;
        INFO() << stats;
        totalStats += stats;
    }

    INFO() << "Total statistics:";
    INFO() << totalStats;

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
