import os
from subprocess import call
import tarfile
import nirvana.job_context as nv

def get_param(name):
    return nv.context().get_parameters().get(name)

def get_input(name):
    return nv.context().get_inputs().get(name)

def get_output(name):
    return nv.context().get_outputs().get(name)

def pack_tarfile(output_filename, filename_list):
    with tarfile.open(output_filename, "w:gz") as tar:
        for source_file in filename_list:
            tar.add(source_file, arcname=os.path.basename(source_file))

def unpack_tarfile(input_filename, source_dir = "."):
    with tarfile.open(input_filename, "r:gz") as tar:
        tar.extractall(path = source_dir)

def get_data_list_file():
    data_folder = "./data"
    if not os.path.exists(data_folder):
        os.mkdir(data_folder)
    images_tar = get_input("images")
    unpack_tarfile(images_tar, data_folder)

    markup_tar = get_input("markup")
    unpack_tarfile(markup_tar, data_folder)

    data_filename = "data.txt"
    data = open(data_filename, "w")

    image_files  = [name[:-4] for name in os.listdir(data_folder) if name.endswith(".jpg")]
    markup_files = [name[:-4] for name in os.listdir(data_folder) if name.endswith(".txt")]
    for name in image_files:
        if name in markup_files:
            data.write("./data/{}.jpg ./data/{}.txt\n".format(name, name))
    data.close()

    return data_filename

def print_params():
    if get_param("data_type") == "segm":
        print("Data type: segmentation")
        print("    Segmetation: {}".format(get_param("segm_dat")))
    elif get_param("data_type") == "edge":
        print("Data type: edge detection")
        print("    Edges: {}".format(get_param("edge_dat")))
    elif get_param("data_type") == "edge_and_vertex":
        print("Data type: edge and vertex detection")
        print("    Edges: {}".format(get_param("edge_dat")))
        print("    Vertices: {}".format(get_param("vertex_dat")))
    # common params
    print("    Images: {}".format(get_param("img_dat")))
    print("Image width: {}".format(get_param("image_width")))
    print("Image hehgt: {}".format(get_param("image_height")))
    print("Equalize histogram: {}".format(get_param("equalize_hist")))
    print("Check not empty: {}".format(get_param("check_not_empty")))
    print("Refine building markup: {}".format(get_param("refine_bld_markup")))
    print(type(get_param("check_not_empty")))

def prepare_segmentation():
    print_params()

    exec_name = get_input("rasterizer")
    data_list_filename = get_data_list_file()

    # Prepare params for command line
    params =  [exec_name]
    params += ["--input_list", data_list_filename]
    params += ["--ann_map", "segm_map.txt"]
    params += ["--img_eq_hist", "true"]
    if get_param("check_not_empty") == True:
        params += ["--check_not_empty",  "true"]
    if get_param("refine_bld_markup") == True:
        params += ["--refine_markup_classes", "bld"]
    params += ["--output_width", str(get_param("image_width"))]
    params += ["--output_height", str(get_param("image_height"))]
    params += ["--output_img_dat", str(get_param("img_dat"))]
    params += ["--output_ann_dat", str(get_param("segm_dat"))]

    # Call rasterizer
    call(params)

    # Pack dataset
    pack_tarfile(get_output("dataset"), [get_param("img_dat"), get_param("segm_dat")])

def prepare_edges():
    print_params()

    exec_name = get_input("rasterizer")
    data_list_filename = get_data_list_file()

    # Prepare params for command line
    params =  [exec_name]
    params += ["--input_list", data_list_filename]
    params += ["--ann_map", "edge_map.txt"]
    params += ["--img_eq_hist", "true"]
    if get_param("check_not_empty") == True:
        params += ["--check_not_empty",  "true"]
    if get_param("refine_bld_markup") == True:
        params += ["--refine_markup_classes", "bld"]
    params += ["--output_width", str(get_param("image_width"))]
    params += ["--output_height", str(get_param("image_height"))]
    params += ["--output_img_dat", str(get_param("img_dat"))]
    params += ["--output_ann_dat", str(get_param("edge_dat"))]

    # Call rasterizer
    call(params)

    # Pack dataset
    pack_tarfile(get_output("dataset"), [get_param("img_dat"), get_param("edge_dat")])

def prepare_edges_and_vertices():
    print_params()

    exec_name = get_input("rasterizer")
    data_list_filename = get_data_list_file()

    # Prepare params for command line
    edge_params =  [exec_name]
    edge_params += ["--input_list", data_list_filename]
    edge_params += ["--ann_map", "edge_map.txt"]
    edge_params += ["--img_eq_hist", "true"]
    if get_param("check_not_empty") == True:
        edge_params += ["--check_not_empty",  "true"]
    if get_param("refine_bld_markup") == True:
        edge_params += ["--refine_markup_classes", "bld"]
    edge_params += ["--output_width", str(get_param("image_width"))]
    edge_params += ["--output_height", str(get_param("image_height"))]
    edge_params += ["--output_img_dat", str(get_param("img_dat"))]
    edge_params += ["--output_ann_dat", str(get_param("edge_dat"))]

    # Call rasterizer
    call(edge_params)

    # Prepare params for command line
    vertex_params =  [exec_name]
    vertex_params += ["--input_list", data_list_filename]
    vertex_params += ["--ann_map", "vertex_map.txt"]
    vertex_params += ["--img_eq_hist", "true"]
    if get_param("check_not_empty") == True:
        vertex_params += ["--check_not_empty",  "true"]
    if get_param("refine_bld_markup") == True:
        vertex_params += ["--refine_markup_classes", "bld"]
    vertex_params += ["--output_width", str(get_param("image_width"))]
    vertex_params += ["--output_height", str(get_param("image_height"))]
    vertex_params += ["--output_img_dat", str(get_param("img_dat"))]
    vertex_params += ["--output_ann_dat", str(get_param("vertex_dat"))]

    # Call rasterizer
    call(vertex_params)

    # Pack dataset
    pack_tarfile(get_output("dataset"), [get_param("img_dat"), get_param("edge_dat"), get_param("vertex_dat")])

def main():
    print("Prepare dataset")
    if get_param("data_type") == "segm":
        prepare_segmentation()
    elif get_param("data_type") == "edge":
        prepare_edges()
    elif get_param("data_type") == "edge_and_vertex":
        prepare_edges_and_vertices()

if __name__ == "__main__":
    main()
