#include "save_type.h"
#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"

#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/exception.h>
#include <yandex/maps/wiki/tasks/branch.h>

namespace maps {
namespace wiki {

namespace {

DECLARE_ERR_CODE( ERR_BRANCH_CANT_DELETE_TRUNK );
DECLARE_ERR_CODE( ERR_BRANCH_CANT_DELETE_APPROVED );
DECLARE_ERR_CODE( ERR_BRANCH_STABLE_ALREADY_EXISTS );
DECLARE_ERR_CODE( ERR_BRANCH_CANT_PUBLISH_NON_STABLE );
const std::string TASK_METHOD_NAME = "SaveBranchType";

void
throwPublicationError(const revision::Branch& branch)
{
    THROW_WIKI_LOGIC_ERROR(
        ERR_BRANCH_CANT_PUBLISH_NON_STABLE,
        "publication failed, type: '" << branch.type() <<
        "' branch: " << branch.id() << "(" << branch.state() << ")");
}

revision::BranchType
convertBranchType(const std::string& branchType, TBranchId branchId)
{
    CHECK_NON_EMPTY_REQUEST_PARAM(branchType);

    if (branchType != BRANCH_TYPE_PRODUCTION) {
        try {
            return boost::lexical_cast<revision::BranchType>(branchType);
        }
        catch (...) {
            THROW_WIKI_LOGIC_ERROR(
                ERR_BAD_REQUEST,
                "invalid parameter type: '" << branchType <<
                "' branch: " << branchId);
        }
    }
    return revision::BranchType::Archive;
}

} // namespace

SaveBranchType::Request::Request(
        TUid user_,
        const std::string& branchType_,
        TBranchId branchId_)
    : user(user_)
    , branchType(convertBranchType(branchType_, branchId_))
    , branchId(branchId_)
{
    WIKI_REQUIRE(
        branchType != revision::BranchType::Stable,
        ERR_BAD_REQUEST,
        "forbidden creating stable branch");

    WIKI_REQUIRE(
        branchType == revision::BranchType::Deleted ||
        branchType == revision::BranchType::Archive,
        ERR_BAD_REQUEST,
        "invalid parameters, type: '" << branchType <<
        "' branch: " << branchId);
}

std::string
SaveBranchType::Request::dump() const
{
    std::stringstream ss;
    ss << " uid: " << user;
    ss << " type: " << branchType;
    ss << " branch: " << branchId;
    return ss.str();
}

SaveBranchType::SaveBranchType(
        const ObserverCollection&,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<SaveBranchType>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(request)
{}

std::string
SaveBranchType::printRequest() const
{
    return request_.dump();
}

void
SaveBranchType::publishBranch()
{
    auto branchCtx = BranchContextFacade(request_.branchId)
        .acquireManageBranches(request_.user);

    try {
        tasks::publishBranch(
            branchCtx.txnCore(), branchCtx.txnLabels(), branchCtx.branch, request_.user);

        result_->branches = loadHeadBranches(branchCtx.txnCore());
        result_->token = branchCtx.commit();
    }
    catch(const maps::wiki::LogicException&) {
        throw;
    }
    catch(...) {
        throwPublicationError(branchCtx.branch);
    }
}

void
SaveBranchType::deleteBranch()
{
    auto branchCtx = BranchContextFacade(request_.branchId)
        .acquireManageBranches(request_.user);

    auto branchType = branchCtx.branch.type();
    if (branchType == revision::BranchType::Trunk) {
        THROW_WIKI_LOGIC_ERROR(
            ERR_BRANCH_CANT_DELETE_TRUNK,
            "can not delete trunk branch");
    }
    if (branchType == revision::BranchType::Approved) {
        THROW_WIKI_LOGIC_ERROR(
            ERR_BRANCH_CANT_DELETE_APPROVED,
            "can not delete approved branch");
    }

    auto& poolLabels = cfg()->poolLabels(branchCtx.branch.id());

    tasks::deleteBranch(
        branchCtx.txnCore(), branchCtx.txnView(), branchCtx.branch, request_.user);

    result_->branches = loadHeadBranches(branchCtx.txnCore());
    result_->token = branchCtx.commit();

    auto workLabels = poolLabels.masterWriteableTransaction();
    tasks::deleteVrevisionsForBranch(*workLabels, branchCtx.branch);
    workLabels->commit();
}

void
SaveBranchType::control()
{
    if (request_.branchType == revision::BranchType::Deleted) {
        deleteBranch();
    } else {
        ASSERT(request_.branchType == revision::BranchType::Archive);
        publishBranch();
    }
}

const std::string&
SaveBranchType::taskName()
{
    return TASK_METHOD_NAME;
}

} // namespace wiki
} // namespace maps
