#include "commit_diff_data_provider.h"
#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"

#include <yandex/maps/wiki/configs/editor/categories.h>

namespace maps {
namespace wiki {

CommitDiffDataProvider::CommitDiffDataProvider(
        const std::unique_ptr<ObjectsCache>& prevCache,
        ObjectsCache& curCache,
        const boost::optional<revision::CommitDiff>& commitDiff)
    : prevCache_(prevCache)
    , curCache_(curCache)
    , commitDiff_(commitDiff)
{
}

const Category&
CommitDiffDataProvider::objectCategory(TOid oid) const
{
    auto cur = curCache_.getExisting(oid);
    return cur->category();
}

bool
CommitDiffDataProvider::isSlaveOf(TOid slaveOid, TOid oid) const
{
    auto slave = curCache_.getExisting(slaveOid);
    for (const auto& master : slave->masterRelations().range()) {
        if (master.id() == oid) {
            return true;
        }
    }
    return false;
}

bool
CommitDiffDataProvider::isIndirectSlaveOf(TOid slaveOid, TOid oid) const
{
    auto slave = curCache_.getExisting(slaveOid);
    for (const auto& master : slave->masterRelations().range()) {
        for (const auto& masterMaster : master.relative()->masterRelations().range()) {
            if (masterMaster.id() == oid) {
                return true;
            }
        }
    }
    return false;
}

TOIds
CommitDiffDataProvider::geomModifiedObjects() const
{
    if (!commitDiff_) {
        return {};
    }
    TOIds objects;
    for (const auto& diff : *commitDiff_) {
        if (diff.second.data().geometry) {
            objects.insert(diff.first);
        }
    }
    return objects;
}

std::vector<RelData>
CommitDiffDataProvider::modifiedRelations() const
{
    if (!commitDiff_) {
        return {};
    }
    std::vector<RelData> relUpdateData;
    TOIds mastersPrefetch;
    for (const auto& diff : *commitDiff_) {
        const auto& newRelData = diff.second.data().newRelationData;
        const auto& oldRelData = diff.second.data().oldRelationData;
        if (oldRelData) {
            mastersPrefetch.insert(oldRelData->masterObjectId());
            relUpdateData.push_back({oldRelData->slaveObjectId(), oldRelData->masterObjectId(), true});
        } else if (newRelData) {
            mastersPrefetch.insert(newRelData->masterObjectId());
            relUpdateData.push_back({newRelData->slaveObjectId(), newRelData->masterObjectId(), false});
        }
    }
    curCache_.get(mastersPrefetch);
    if (prevCache_) {
        prevCache_->get(mastersPrefetch);
    }
    return relUpdateData;
}

Geom
CommitDiffDataProvider::oldGeometry(TOid oid) const
{
    if (commitDiff_) {
        auto it = commitDiff_->find(oid);
        if (it != commitDiff_->end() && it->second.data().geometry) {
            const auto& wkb = it->second.data().geometry->before;
            return wkb.empty() ? Geom() : Geom(wkb);
        }
    }
    if (!prevCache_) {
        return Geom();
    }
    auto obj = prevCache_->get(oid);
    return (obj && !(*obj)->isDeleted()) ? (*obj)->geom() : Geom();
}

Geom
CommitDiffDataProvider::newGeometry(TOid oid) const
{
    if (commitDiff_) {
        auto it = commitDiff_->find(oid);
        if (it != commitDiff_->end() && it->second.data().geometry) {
            const auto& wkb = it->second.data().geometry->after;
            return wkb.empty() ? Geom() : Geom(wkb);
        }
    }
    auto obj = curCache_.getExisting(oid);
    return !obj->isDeleted() ? obj->geom() : Geom();
}

TOIds
CommitDiffDataProvider::objectGeomSlaves(TOid oid) const
{
    RelativesLimit slavesLimit {SLAVES_LIMIT_MULTIPLIER * cfg()->editor()->system().slavesPerRoleLimit(), RelativesLimit::PerRole};
    auto obj =
        curCache_.getExisting(oid)->isDeleted() && prevCache_
            ? prevCache_->getExisting(oid)
            : curCache_.getExisting(oid);
    auto ranges = *(obj->relations(RelationType::Slave).range(slavesLimit));
    auto rolesWithExceededLimit = ranges.rolesWithExceededLimit();
    TOIds  slaves;
    const StringSet geomRoles = obj->category().slaveRoleIds(roles::filters::IsGeom);
    for (const auto& roleId :  geomRoles) {
        if (rolesWithExceededLimit.count(roleId) > 0) {
            throw DiffLimitExceeded();
        }
        for (const auto& slave : obj->slaveRelations().range(roleId)) {
            slaves.insert(slave.id());
        }
    }
    return slaves;
}

} // namespace wiki
} // namespace maps
