#include "get_commit_geom_diff.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/object.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/junction.h"
#include "maps/wikimap/mapspro/services/editor/src/commit.h"
#include "maps/wikimap/mapspro/services/editor/src/objects_cache.h"
#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/category_traits.h"
#include <maps/libs/tile/include/utils.h>

namespace maps {
namespace wiki {
namespace {
const size_t MAX_GEOM_DIFF_POINTS_NUMBER = 2000;
const double START_GEOMETRY_SIMPLIFICATION_TOLERANCE_RATIO = 3.0;
const double NEXT_GEOMETRY_SIMPLIFICATION_TOLERANCE_RATIO = 1.5;
const CachePolicy CACHE_POLICY
{
    TableAttributesLoadPolicy::Load,
    ServiceAttributesLoadPolicy::Skip, // default: Load
    DanglingRelationsPolicy::Ignore // default: Check
};
}// namespace

GetCommitGeomDiff::GetCommitGeomDiff(const Request& request)
    : controller::BaseController<GetCommitGeomDiff>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{
    CHECK_REQUEST_PARAM(request.commitId);
}

GetCommitGeomDiff::~GetCommitGeomDiff()
{}

std::string
GetCommitGeomDiff::printRequest() const
{
    std::stringstream ss;

    ss << " commit:" << request_.commitId;
    ss << " object:" << request_.objectId;
    ss << " user: " << request_.user;
    ss << " token: " << request_.dbToken;
    ss << " branch: " << request_.branchId;
    return ss.str();
}

void
GetCommitGeomDiff::control()
{
    auto branchCtx = BranchContextFacade::acquireRead(
        request_.branchId, request_.dbToken);
    ObjectsCache curCache(branchCtx, request_.commitId, CACHE_POLICY);
    RevisionsFacade& tdsGw = curCache.revisionsFacade();
    auto commit = tdsGw.loadHeadCommit();
    const auto& action = commit.action();
    if ((!common::COMMIT_NON_GROUP_ACTIONS.count(action) &&
        !common::COMMIT_GROUP_ACTIONS.count(action))) {
        return;
    }
    if (action == common::COMMIT_PROPVAL_ACTION_IMPORT) {
        if (!request_.objectId) {
            return;
        }
        auto object = curCache.getExisting(request_.objectId);
        result_->geomDiff.tryAddAfter(object->geom());
        return;
    }

    auto diffScope = commitDiffScope(commit, request_.objectId);

    auto commitDiff = revision::commitDiff(
            branchCtx.txnCore(), request_.commitId,
            diffScope == DiffScope::Commit ? 0 : request_.objectId);

    std::unique_ptr<ObjectsCache> prevCache;
    if (request_.commitId > 1) {
        prevCache.reset(new ObjectsCache(
            branchCtx, request_.commitId - 1, CACHE_POLICY));
    }

    result_->geomDiff = objectGeomDiff(
            prevCache, curCache, commitDiff,
            request_.objectId, affectedObjectIds(commit), diffScope,
            createEnvelope(request_.bboxString, SpatialRefSystem::Geodetic),
            DEFAULT_GEOM_DIFF_PARTS_LIMIT);
    auto& geomDiff = result_->geomDiff;
    auto numPoints = geomDiff.numPoints();
    double tolerance =
        START_GEOMETRY_SIMPLIFICATION_TOLERANCE_RATIO
        * tile::zoomToResolution(request_.zoom);
    while (numPoints > MAX_GEOM_DIFF_POINTS_NUMBER) {
        auto simplifiedNumPoints = geomDiff.simplify(tolerance);
        if (simplifiedNumPoints == numPoints) {
            break;
        }
        numPoints = simplifiedNumPoints;
        tolerance *= NEXT_GEOMETRY_SIMPLIFICATION_TOLERANCE_RATIO;
    }
}

} // namespace wiki
} // namespace maps
