#include "revert.h"
#include "reverter.h"
#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <maps/wikimap/mapspro/services/editor/src/moderation_log.h>
#include <maps/wikimap/mapspro/services/editor/src/actions/social/moderation/helpers.h>

#include <yandex/maps/wiki/social/gateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>

#include <maps/wikimap/mapspro/libs/acl_utils/include/moderation.h>

namespace maps::wiki {

namespace {

const std::string TASK_METHOD_NAME = "CommitsRevert";

class OwnCommitsReverter : public CommitsReverter
{
public:
    OwnCommitsReverter(
            const ObserverCollection& observers,
            BranchContext& branchCtx,
            UserContext& userContext,
            boost::optional<TId> feedbackTaskId)
        : CommitsReverter(observers, branchCtx, userContext, std::move(feedbackTaskId))
    {
        const auto& status = userContext.moderationStatus(branchCtx);

        isModerator_ = acl_utils::isModerator(status);
        isCartographer_ = acl_utils::isCartographer(status);
        cacheCartographerState_.insert({uid(), isCartographer_});
    }

    bool isCartographer(TUid uid) const
    {
        auto it = cacheCartographerState_.find(uid);
        if (it != cacheCartographerState_.end()) {
            return it->second;
        }

        acl::ACLGateway gateway(branchCtx().txnCore());
        auto user = gateway.user(uid);

        auto status = acl_utils::moderationStatus(gateway, user);
        auto isCartographer = acl_utils::isCartographer(status);
        cacheCartographerState_.insert({uid, isCartographer});
        return isCartographer;
    }

protected:
    void checkRevertedCommit(
        const revision::Commit& commit,
        const TCommitIds& allRevertedCommitIds) const override
    {
        WIKI_REQUIRE(
            isCartographer_ || commit.state() == revision::CommitState::Draft,
            ERR_REVERT_NOT_DRAFT_COMMIT,
            "Can not revert already approved commit " << commit.id());

        auto createdBy = commit.createdBy();
        if (!isCartographer_ && createdBy != uid()) {
            WIKI_REQUIRE(
                isModerator_ && !isCartographer(createdBy),
                ERR_REVERT_NOT_OWNER,
                "Can not revert commit " << commit.id() << ", created by another user"
                " (" << createdBy << " != " << uid() << ")");
        }
        CommitsReverter::checkRevertedCommit(commit, allRevertedCommitIds);
    }

private:
    bool isModerator_ = false;
    bool isCartographer_ = false;

    mutable std::map<TUid, bool> cacheCartographerState_;
};

} // namespace

CommitsRevert::Request::Request(
    UserContext userContext,
    TCommitId commitId,
    boost::optional<RevertReason> revertReason,
    boost::optional<TId> feedbackTaskId
)
    : userContext(std::move(userContext))
    , commitId(commitId)
    , revertReason(std::move(revertReason))
    , feedbackTaskId(std::move(feedbackTaskId))
{
    CHECK_REQUEST_PARAM(commitId);
}

std::string
CommitsRevert::Request::dump() const
{
    std::stringstream ss;
    ss << " uid: " << userId()
       << " commit: " << commitId;
    if (revertReason) {
        ss << " revert reason: " << boost::lexical_cast<std::string>(*revertReason);
    }
    if (feedbackTaskId) {
        ss << " feedback-task-id: " << *feedbackTaskId;
    }
    return ss.str();
}

CommitsRevert::CommitsRevert(
        const ObserverCollection& observers,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<CommitsRevert>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , observers_(observers)
    , request_(request)
{}

std::string
CommitsRevert::printRequest() const
{
    return request_.dump();
}

void
CommitsRevert::control()
{
    processRevert();
    logModerationEvent(result_->taskIds, ModerationLogger::Action::Closed);
}

void
CommitsRevert::processRevert()
{
    auto branchViewContext =
        BranchContextFacade::acquireWrite(revision::TRUNK_BRANCH_ID, request_.userId());

    OwnCommitsReverter reverter(
        observers_, branchViewContext, request_.userContext, request_.feedbackTaskId);
    const auto revertData = reverter.revertCommits({request_.commitId}, request_.revertReason);

    social::Gateway socialGw(branchViewContext.txnSocial());
    const auto aoisCommitBelongsTo =
        socialGw.getAoiIdsOfActiveEditTask(request_.commitId);

    result_->taskIds = moderation::closeOrResolveRevertedTasks(
        branchViewContext,
        socialGw,
        reverter.user(),
        revertData.revertedCommitIds,
        aoisCommitBelongsTo,
        {request_.commitId},
        reverter.isCartographer(request_.userId())
    );

    result_->token = reverter.syncViewAndCommit(revertData);
    result_->commitModel = make_unique<CommitModel>(revertData.createdCommit);
}

const std::string&
CommitsRevert::taskName()
{
    return TASK_METHOD_NAME;
}

} // namespace maps::wiki
