#pragma once

#include <maps/wikimap/mapspro/libs/controller/include/asyncbasecontroller.h>
#include <maps/wikimap/mapspro/libs/controller/include/async_control_support.h>
#include <maps/wikimap/mapspro/services/editor/src/exception_handling.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/formatter.h>
#include <maps/wikimap/mapspro/libs/controller/include/asynctaskcontext.h>
#include <yandex/maps/wiki/common/format_type.h>

#include <memory>

namespace maps::wiki {

class ObserverCollection;

template <typename Controller>
class ControllerAsyncTaskContext : public controller::AsyncTaskContext {
public:
    typedef typename Controller::Request Request;

    ControllerAsyncTaskContext(
            const ObserverCollection& observers,
            taskutils::Task task,
            std::shared_ptr<Request> request,
            common::FormatType formatType)
        : AsyncTaskContext(task)
        , observers_(observers)
        , request_(request)
        , formatType_(formatType)
    {}

protected:
    void createWorker() override
    {
        auto worker = make_unique<Controller>(observers_, *request_, task().id());
        worker_.swap(worker);
    }

    std::string invokeWorker() override
    {
        auto formatter = Formatter::create(formatType_);
        auto result = (*worker_)();
        return (*formatter)(*result);
    }

    ExceptionInfo handleWorkerException() override
    {
        return handleException(
            {},
            formatType_,
            InternalErrorPolicy::Serialize,
            worker_ ? &worker_->profile() : nullptr);
    }

    std::string formatResult(const controller::AsyncTaskResult& asyncTaskResult) const override
    {
        auto formatter = Formatter::create(formatType_);
        return (*formatter)(asyncTaskResult);
    }

private:
    const ObserverCollection& observers_;
    std::shared_ptr<Request> request_;
    common::FormatType formatType_;
    std::unique_ptr<Controller> worker_;
};

template <typename Controller>
class ControllerAsync: public controller::AsyncBaseController
{
public:
    typedef typename Controller::Request Request;

    ControllerAsync(
            const ObserverCollection& observers,
            const Request& request,
            common::FormatType formatType,
            WaitPolicy waitPolicy = WaitPolicy::Timeout)
        : controller::AsyncBaseController(BOOST_CURRENT_FUNCTION, waitPolicy, cfg()->asyncTasksSupport())
        , observers_(observers)
        , request_(std::make_shared<Request>(request))
        , formatType_(formatType)
    {}

    const std::string& taskName() const override { return Controller::taskName(); }

    std::string printRequest() const override { return request_->dump(); }


protected:
    void
    createAndExecTask()
    {
        auto taskControlData = asyncTaskControlData<Controller>(*request_);
        auto task = createTask(
            request_->userId(),
            taskControlData);
        ASSERT (taskControlData.empty() || task);
        DEBUG() << "ControllerAsync<" << taskName() << "> is recoverable: " << !taskControlData.empty();
        if (task) {
            execTaskContext(
                std::make_shared<ControllerAsyncTaskContext<Controller>>(
                    observers_,
                    *task,
                    request_,
                    formatType_));
        }
    }
    void control() override
    {
        createAndExecTask();
    }

private:
    const ObserverCollection& observers_;
    std::shared_ptr<Request> request_;
    common::FormatType formatType_;
};

} // namespace maps::wiki
