#include "distribute_flats.h"
#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/common.h>

#include <maps/wikimap/mapspro/libs/flat_range/include/distribution.h>
#include <maps/wikimap/mapspro/libs/flat_range/include/range.h>
#include <maps/wikimap/mapspro/libs/flat_range/include/validation.h>

#include <yandex/maps/wiki/common/string_utils.h>
#include <maps/libs/json/include/value.h>

namespace maps::wiki {

namespace {

const std::string TASK_METHOD_NAME = "DistributeFlats";

DECLARE_ERR_CODE( ERR_BAD_FLAT_RANGE );
DECLARE_ERR_CODE( ERR_BAD_LEVEL_RANGE );
DECLARE_ERR_CODE( ERR_TOO_MANY_FLATS_PER_ENTRANCE );
DECLARE_ERR_CODE( ERR_FLAT_RANGES_INTERSECTION );
DECLARE_ERR_CODE( WARN_INCOMPLETE_DISTRIBUTION );

const std::map<flat_range::ValidationResult, std::string> PROBLEM_TO_ERROR {
    {flat_range::ValidationResult::BadFlatRange, ERR_BAD_FLAT_RANGE},
    {flat_range::ValidationResult::BadLevelRange, ERR_BAD_LEVEL_RANGE},
    {flat_range::ValidationResult::FlatRangesIntersection, ERR_FLAT_RANGES_INTERSECTION},
    {flat_range::ValidationResult::TooManyFlatsPerEntrance, ERR_TOO_MANY_FLATS_PER_ENTRANCE},
};

} // namespace

DistributeFlats::DistributeFlats(const Request& request)
    : controller::BaseController<DistributeFlats>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{
}

const std::string&
DistributeFlats::taskName()
{
    return TASK_METHOD_NAME;
}

std::string
DistributeFlats::Request::dump() const
{
    std::stringstream ss;
    ss << body;
    return ss.str();
}

std::string
DistributeFlats::printRequest() const
{
    return request_.dump();
}

void
DistributeFlats::control()
{
    std::vector<flat_range::FlatLevelRange> flatLevelRanges;

    const auto json = json::Value::fromString(request_.body);
    WIKI_REQUIRE(
        !json.empty()
        && json.hasField(STR_FLAT_RANGES)
        && json[STR_FLAT_RANGES].isArray(),
        ERR_BAD_DATA,
        "Expected array of flat/level ranges");

    for (const auto& objValue : json[STR_FLAT_RANGES]) {
        WIKI_REQUIRE(
            objValue.hasField(STR_FLATS),
            ERR_BAD_DATA,
            "Expected flats and levels in range");
        flatLevelRanges.push_back({
            objValue[STR_FLATS].as<std::string>(),
            objValue[STR_LEVELS].as<std::string>("")});
    }

    auto validationResult = flat_range::validate(flatLevelRanges);
    auto distributionResult = flat_range::distributeFlatsByLevels(
        flatLevelRanges);
    result_->levelsWithFlats = distributionResult.levelsWithFlats;

    for (const auto& problem : validationResult) {
        if (PROBLEM_TO_ERROR.contains(problem)) {
            result_->error = PROBLEM_TO_ERROR.at(problem);
            break;
        }
    }
    if (result_->error.empty() && !distributionResult.isComplete) {
        result_->warning = WARN_INCOMPLETE_DISTRIBUTION;
    }
    result_->levelsWithFlats = distributionResult.levelsWithFlats;
}

} // namespace maps::wiki
