#include "get_topology.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/object.h"
#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"
#include "maps/wikimap/mapspro/services/editor/src/branch_helpers.h"
#include "maps/wikimap/mapspro/services/editor/src/common.h"
#include "maps/wikimap/mapspro/services/editor/src/objects_cache.h"

#include <maps/wikimap/mapspro/libs/views/include/query_builder.h>
#include <yandex/maps/wiki/configs/editor/topology_groups.h>

#include <cmath>

namespace maps {
namespace wiki {
namespace {
const size_t BBOX_THRESHOLD = 10;
}

GetTopology::GetTopology(const Request& request)
    : controller::BaseController<GetTopology>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{
}

std::string
GetTopology::printRequest() const
{
    std::stringstream ss;
    ss << " oid: " << request_.oid;
    ss << " token: " << request_.token;
    ss << " bb: " << request_.bbox;
    ss << " branch: " << request_.branchId;
    return ss.str();
}


void
GetTopology::topologyByBBox(const ObjectPtr& object, ObjectsCache& cache)
{
    const auto* editorCfg = cfg()->editor();
    const auto& contourDefs = editorCfg->contourObjectsDefs();
    if ( cfg()->editor()->contourObjectsDefs().partType(object->categoryId()) !=
        ContourObjectsDefs::PartType::Contour) {
        return;
    }
    const auto& contourDef = contourDefs.contourDef(object->categoryId());
    const std::string& linearElementsCategory =
        contourDef.contour.linearElement.categoryId;
    auto topoGrp = editorCfg->topologyGroups().findGroup(linearElementsCategory);
    REQUIRE(topoGrp, "Topology group not found for requested object parts.");
    auto envelope = createEnvelope(request_.bbox, SpatialRefSystem::Geodetic);
    TOIds linearElementsIds;

    auto linearElementsSlaves = object->slaveRelations().range(
        contourDef.contour.linearElement.roleId,
        {BBOX_THRESHOLD,  RelativesLimit::PerRole});
    ASSERT(linearElementsSlaves);
    if (linearElementsSlaves->rolesWithExceededLimit().empty()) {
        for (const auto& linearElementsSlave : *linearElementsSlaves) {
            linearElementsIds.insert(linearElementsSlave.id());
        }
        result_->isComplete = true;
    } else {
        views::QueryBuilder qb(request_.branchId);
        qb.selectFields("l.id");
        qb.fromTable(views::TABLE_OBJECTS_L, "l");
        qb.fromTable(views::TABLE_OBJECTS_R, "r");

        std::stringstream whereClause;
        whereClause.precision(DOUBLE_FORMAT_PRECISION);

        auto& work = cache.workView();
        whereClause <<
            " l.id=r.slave_id AND "
            " r.master_id=" << request_.oid << " AND "
            " l.domain_attrs ? " <<
                work.quote(plainCategoryIdToCanonical(linearElementsCategory)) << " AND "
            " l.the_geom && ST_SetSRID(ST_MakeBox2D("
            " ST_MakePoint(" << envelope.getMinX() << "," << envelope.getMinY() << "),"
            " ST_MakePoint(" << envelope.getMaxX() << "," << envelope.getMaxY() << ")),3395)";
        if (cfg()->viewPartsEnabled()) {
            whereClause <<
                " AND l.part=ANY(get_parts(" <<
                envelope.getMinX() << "," << envelope.getMinY() << "," <<
                envelope.getMaxX() << "," << envelope.getMaxY() << "))";
        }
        qb.whereClause(whereClause.str());

        for (const auto& row : work.exec(qb.query())) {
            linearElementsIds.insert(row[0].as<TOid>());
        }
    }

    if (linearElementsIds.empty()) {
        return;
    }
    if (!result_->isComplete && !linearElementsIds.empty()) {
        size_t linearElementsCount = linearElementsIds.size();
        auto moreLinearElements = object->slaveRelations().range(
            contourDef.contour.linearElement.roleId,
            {linearElementsCount,  RelativesLimit::PerRole});
        result_->isComplete = moreLinearElements &&
            moreLinearElements->size() == linearElementsCount;
    }
    StringSet junctionsRoles{topoGrp->startJunctionRole(), topoGrp->endJunctionRole()};
    result_->linearElements = cache.get(linearElementsIds);
    TOIds junctionsOutOfScopeIds;
    for (const auto& linearElement : result_->linearElements) {
        const auto& junctionsRelationsRange = linearElement->slaveRelations().range(junctionsRoles);
        for (const auto& junctionRelation : junctionsRelationsRange) {
            auto junction = cache.getExisting(junctionRelation.id());
            if (result_->isComplete || envelope.contains(junction->geom()->getEnvelopeInternal())) {
                result_->objectsLinks[junction->id()].insert({junctionRelation.roleId(), linearElement.get()});
                result_->junctions.add(junction);
            }
            result_->objectsLinks[linearElement->id()].insert({junctionRelation.roleId(), junction.get()});
        }
    }
    for (const auto& junction : result_->junctions) {
        for (const auto& linearElementRelation : junction->masterRelations().range(junctionsRoles)) {
            TOid adjacentId = linearElementRelation.id();
            if (linearElementsIds.count(adjacentId)) {
                continue;
            }
            result_->adjacentLinearElements.add(cache.getExisting(adjacentId));
            result_->objectsLinks[adjacentId].insert({linearElementRelation.roleId(), junction.get()});
            result_->objectsLinks[junction->id()].insert({linearElementRelation.roleId(),
                linearElementRelation.relative()});
        }
    }
}

void
GetTopology::control()
{
    auto branchCtx = BranchContextFacade::acquireRead(request_.branchId, request_.token);
    result_->cache.reset(new ObjectsCache(
             branchCtx, boost::none,
             { TableAttributesLoadPolicy::Skip,
               ServiceAttributesLoadPolicy::Skip,
               DanglingRelationsPolicy::Check }));
    auto contourObj = result_->cache->getExisting(request_.oid);
    if (contourObj->isDeleted()) {
        return;
    }
    topologyByBBox(contourObj, *result_->cache);
}

} // namespace wiki
} // namespace maps
