#include "getslaveinfos.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/object.h"
#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"
#include "maps/wikimap/mapspro/services/editor/src/branch_helpers.h"
#include "maps/wikimap/mapspro/services/editor/src/common.h"
#include "maps/wikimap/mapspro/services/editor/src/factory.h"

#include <maps/wikimap/mapspro/libs/dbutils/include/parser.h>
#include <maps/wikimap/mapspro/libs/views/include/query_builder.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <maps/libs/log8/include/log8.h>
#include <contrib/libs/fmt/include/fmt/format.h>

#include <cmath>

namespace maps {
namespace wiki {

namespace {
const std::string LESS = "<";
const std::string GREATER = ">";
const std::string SLAVE_ID_ORDER = " ORDER BY slave.id ASC ";
const std::string SLAVE_ID_ORDER_REVERT = " ORDER BY slave.id DESC ";

const std::string SLAVE_DATA =
    " slave.id as slave_id,"
    " relation.id as relation_id,"
    " slave.commit_id as slave_commit_id,"
    " slave.service_attrs->'srv:screen_label' as slave_screen_label,"
    " relation.domain_attrs->'rel:role' as slave_role,"
    " relation.domain_attrs->'rel:slave' as slave_category,"
    " hstore_to_array(slave.service_attrs) as slave_service_attrs";

const std::string COMPLEX_SLAVE_DATA =
    " NULL::geometry as slave_geometry," + SLAVE_DATA;

const std::string GEOM_SLAVE_DATA =
    " slave.the_geom as slave_geometry," + SLAVE_DATA;

const std::string MASTER_ID_AND_ROLES_CONDITION =
    " AND relation.master_id={0} "
    " AND relation.domain_attrs->'rel:role' = {1}"
    " AND relation.domain_attrs->'rel:slave' = {2}";
const std::string BOUNDING_BOX =
    " ST_SetSRID(ST_MakeBox2D("
            " ST_MakePoint({3:.12f}, {4:.12f}),"
            " ST_MakePoint({5:.12f}, {6:.12f})),3395)";
const std::string SLAVE_ID_JOIN =
    " slave.id=relation.slave_id ";

enum class CollectGeom
{
    Yes,
    No
};

controller::ResultType<GetSlaveInfos>::SlaveObjectInfo
slaveInfoFromRow(const pqxx::row& row, CollectGeom collectGeom)
{
    return {
        row["slave_id"].as<TOid>(),
        row["slave_role"].c_str(),
        row["relation_id"].as<TOid>(),
        TRevisionId(
            row["slave_id"].as<TOid>(),
            row["slave_commit_id"].as<TCommitId>()),
        row["slave_screen_label"].c_str(),
        row["slave_category"].c_str(),
        collectGeom == CollectGeom::Yes ? Geom(row["slave_geometry"]) : Geom(),
        0,
        dbutils::parsePGHstore(row["slave_service_attrs"].c_str())
    };
}

template<typename RowIter>
std::vector<controller::ResultType<GetSlaveInfos>::SlaveObjectInfo>
collectResult(RowIter begin, RowIter end, CollectGeom collectGeom, size_t limit)
{
    std::vector<controller::ResultType<GetSlaveInfos>::SlaveObjectInfo> slaveInfos;
    for (auto it = begin; it != end && (limit == 0 || slaveInfos.size() < limit); ++it) {
        slaveInfos.emplace_back(slaveInfoFromRow(*it, collectGeom));
    }
    return slaveInfos;
}

} // namespace


const std::string&
controller::ResultType<GetSlaveInfos>::SlaveObjectInfo::serviceAttrValue(const std::string& attrName) const
{
    auto it = serviceAttrs_.find(attrName);
    return it == serviceAttrs_.end()
        ? s_emptyString
        : it->second;
}

GetSlaveInfos::GetSlaveInfos(const Request& request)
    : controller::BaseController<GetSlaveInfos>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{
    result_->roleId = request_.roleId;
    result_->limit = request_.limit;
    result_->offset = request_.offset;
    result_->bbox = request_.bbox;
}

std::string
GetSlaveInfos::printRequest() const
{
    std::stringstream ss;
    ss << " oid: " << request_.oid;
    ss << " token: " << request_.token;
    ss << " role-id: " << request_.roleId;
    ss << " category-id: " << request_.categoryId;
    ss << " bb: " << request_.bbox;
    ss << " offset: " << request_.offset;
    ss << " limit: " << request_.limit;
    ss << " branch: " << request_.branchId;
    return ss.str();
}

void
GetSlaveInfos::slavesByBBox(
    Transaction& work,
    const Category& slaveCategory)
{
    bool slavesHasGeometry = !slaveCategory.complex();

    views::QueryBuilder qb(request_.branchId);
    qb.selectFields(slavesHasGeometry ? GEOM_SLAVE_DATA : "DISTINCT " + COMPLEX_SLAVE_DATA);
    qb.fromTable(views::TABLE_OBJECTS_R, "relation");
    qb.fromTable(GeoObjectFactory::objectClass(slaveCategory.id()).tableName, "slave");

    if (slavesHasGeometry) {
        qb.whereClause(
            SLAVE_ID_JOIN + MASTER_ID_AND_ROLES_CONDITION +
            " AND (slave.the_geom && " + BOUNDING_BOX +")");
    } else {
        qb.fromTable(views::TABLE_OBJECTS_R, "relation_next");
        qb.fromTable(views::TABLE_OBJECTS_L, "slave_next");
        qb.whereClause(
            SLAVE_ID_JOIN + MASTER_ID_AND_ROLES_CONDITION +
            " AND (slave_next.the_geom && " + BOUNDING_BOX + ") "
            " AND slave_next.id=relation_next.slave_id "
            " AND relation_next.master_id=relation.slave_id");
    }

    const std::string queryFMT = qb.query();

    auto envelope = createEnvelope(request_.bbox, SpatialRefSystem::Geodetic);
    auto queryResult = work.exec(fmt::format(queryFMT,
        request_.oid,
        work.quote(request_.roleId),
        work.quote(slaveCategory.id()),
        envelope.getMinX(),
        envelope.getMinY(),
        envelope.getMaxX(),
        envelope.getMaxY()));
    if (!queryResult.empty()) {
        result_->slaveInfos = collectResult(
            queryResult.begin(),
            queryResult.end(),
            slavesHasGeometry ? CollectGeom::Yes : CollectGeom::No,
            0);
    }
}

void
GetSlaveInfos::slavesByOffsetAndLimit(
    Transaction& work,
    const Category& slaveCategory)
{
    bool slavesHasGeometry = !slaveCategory.complex();

    views::QueryBuilder qb(request_.branchId);
    qb.selectFields(slavesHasGeometry ? GEOM_SLAVE_DATA : COMPLEX_SLAVE_DATA);
    qb.fromTable(views::TABLE_OBJECTS_R, "relation");
    qb.fromTable(GeoObjectFactory::objectClass(slaveCategory.id()).tableName, "slave");
    qb.whereClause(SLAVE_ID_JOIN + MASTER_ID_AND_ROLES_CONDITION);

    std::string query = qb.query();
    if (request_.format == common::FormatType::XML) {
        const std::string queryFMT = query + " ORDER BY slave.id OFFSET {3} LIMIT {4}";

        auto queryResult = work.exec(fmt::format(queryFMT,
            request_.oid,
            work.quote(request_.roleId),
            work.quote(slaveCategory.id()),
            request_.offset,
            request_.limit
        ));
        result_->slaveInfos = collectResult(
            queryResult.begin(),
            queryResult.end(),
            CollectGeom::Yes,
            request_.limit);
    } else {
        query +=
            " AND slave.id "
            + (request_.beforeAfter == common::BeforeAfter::Before ? LESS : GREATER)
            + " {3} ";

        bool reverse = request_.startId && request_.beforeAfter == common::BeforeAfter::Before;
        query += reverse ? SLAVE_ID_ORDER_REVERT : SLAVE_ID_ORDER;
        query += " LIMIT {4}";
        auto queryResult = work.exec(fmt::format(query,
            request_.oid,
            work.quote(request_.roleId),
            work.quote(slaveCategory.id()),
            request_.startId,
            request_.limit + 1));
        result_->hasMore = request_.limit ? (queryResult.size() > request_.limit) : false;
        result_->slaveInfos = reverse
            ? collectResult(
                request_.limit >= queryResult.size() ? queryResult.rbegin() : queryResult.rbegin() + 1,
                queryResult.rend(),
                CollectGeom::Yes,
                request_.limit)
            : collectResult(
                queryResult.begin(),
                queryResult.end(),
                CollectGeom::Yes,
                request_.limit);
    }
}

void
GetSlaveInfos::control()
{
    const auto& categories = cfg()->editor()->categories();
    WIKI_REQUIRE(
        categories.defined(request_.categoryId),
        ERR_BAD_REQUEST,
        "Unknow category: '" << request_.categoryId << "'");

    const auto& slaveCategory = categories[request_.categoryId];
    if (!slaveCategory.syncView()) {
        return;
    }

    auto work = BranchContextFacade::acquireWorkReadViewOnly(
        request_.branchId, request_.token);

    if (!request_.bbox.empty()) {
        slavesByBBox(*work, slaveCategory);
    } else {
        slavesByOffsetAndLimit(*work, slaveCategory);
    }
}

} // namespace wiki
} // namespace maps
