#include "objects_cleanup_relations.h"

#include "maps/wikimap/mapspro/services/editor/src/branch_helpers.h"
#include "maps/wikimap/mapspro/services/editor/src/commit.h"
#include "maps/wikimap/mapspro/services/editor/src/objects_cache.h"
#include "maps/wikimap/mapspro/services/editor/src/relations_manager.h"
#include "maps/wikimap/mapspro/services/editor/src/relations_processor.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/relation_object.h"

#include <maps/wikimap/mapspro/libs/acl_utils/include/moderation.h>

namespace maps {
namespace wiki {

namespace {

const std::string TASK_METHOD_NAME = "ObjectsCleanupRelations";

} // namespace

ObjectsCleanupRelations::ObjectsCleanupRelations(
        const ObserverCollection& observers,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<ObjectsCleanupRelations>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , observers_(observers)
    , request_(request)
{}

std::string
ObjectsCleanupRelations::printRequest() const
{
    return request_.dump();
}

std::string
ObjectsCleanupRelations::Request::dump() const
{
    std::stringstream ss;
    ss << " user: " << userId();
    ss << " oid: " << objectId;
    ss << " branch: " << branchId;
    return ss.str();
}

void
ObjectsCleanupRelations::control()
{
    auto branchContext = BranchContextFacade::acquireWrite(request_.branchId, request_.userId());

    const auto& moderationStatus = request_.userContext.moderationStatus(branchContext);
    WIKI_REQUIRE(acl_utils::isCartographer(moderationStatus),
        ERR_FORBIDDEN,
        "User " << request_.userId() << " is not cartographer");

    ObjectsCache cache(branchContext, boost::none);
    auto object = cache.getExisting(request_.objectId);

    if (object->isDeleted()) {
        RelationsProcessor processor(cache);
        processor.deleteAllRelations(object.get());
    } else {
        auto deleteRelations = [&](RelationType relationType) {
            auto range = object->relations(relationType).range();
            for (const auto& rel : rangeToInfos(range)) {
                if (rel.relative()->isDeleted()) {
                    cache.relationsManager().deleteRelation(
                        relationType == RelationType::Master ? rel.relative()->id() : object->id(),
                        relationType == RelationType::Slave ? rel.relative()->id() : object->id(),
                        rel.roleId());
                }
            }
        };

        deleteRelations(RelationType::Master);
        deleteRelations(RelationType::Slave);
    }

    saveAndNotify(cache);
}

void
ObjectsCleanupRelations::saveAndNotify(ObjectsCache& cache)
{
    cache.save(request_.userId(), common::COMMIT_PROPVAL_OBJECT_MODIFIED);
    result_->taskName = taskName();
    result_->token = observers_.doCommit(cache, request_.userContext);
    result_->commit = cache.savedCommit();
}

const std::string&
ObjectsCleanupRelations::taskName()
{
    return TASK_METHOD_NAME;
}

} // namespace wiki
} // namespace maps
