#include "objects_query_lasso.h"

#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"
#include "maps/wikimap/mapspro/services/editor/src/branch_helpers.h"
#include "maps/wikimap/mapspro/services/editor/src/utils.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/category_traits.h"
#include "maps/wikimap/mapspro/services/editor/src/srv_attrs/registry.h"
#include "maps/wikimap/mapspro/services/editor/src/views/objects_query.h"

#include <yandex/maps/wiki/filters/stored_expression.h>

namespace maps {
namespace wiki {

ObjectsQueryLasso::ObjectsQueryLasso(const Request& request)
    : controller::BaseController<ObjectsQueryLasso>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{
}

std::string
ObjectsQueryLasso::Request::dump() const
{
    std::stringstream ss;
    ss << " categories[" << categories << "]";
    ss << " geom: " << geomJson;
    ss << " token: " << dbToken;
    ss << " branch: " << branchId;
    ss << " limit: " << limit;
    if (indoorLevelId) {
        ss << " indoor-level-id: " << *indoorLevelId;
    }
    return ss.str();
}

std::string
ObjectsQueryLasso::printRequest() const
{
    return request_.dump();
}

namespace {
StringSet parseCategories(
    const std::string& categoriesString,
    ObjectsQueryLasso::SubstitutionPolicy substitutionPolicy)
{
    const auto categories = split<StringSet>(categoriesString, ',');
    if (substitutionPolicy == ObjectsQueryLasso::SubstitutionPolicy::Deny) {
        return categories;
    }
    const auto& contourDefs = cfg()->editor()->contourObjectsDefs();
    StringSet substituted;
    for (const auto& categoryId : categories) {
        if (!isContourElementCategory(categoryId)) {
            substituted.insert(categoryId);
        } else {
            substituted.insert(
                contourDefs.contourDef(categoryId).categoryId);
        }
    }
    return substituted;
}

views::ViewObjects
enrichWithServiceAttrs(
    const views::ViewObjects& objects,
    Transaction& workView,
    TBranchId branchId)
{
    if (objects.empty()) {
        return objects;
    }
    TOIds ids;
    for (const auto& object : objects) {
        ids.insert(object.id());
    }
    views::ObjectsQuery objectsQuery;
    objectsQuery.addCondition(views::OidsCondition(ids));
    return objectsQuery.exec(workView, branchId);
}

void
sortViewObjects(views::ViewObjects& objects, const Geom& geom) {
    std::sort(objects.begin(), objects.end(),
        [&](const views::ViewObject& l, const views::ViewObject& r) {
            if (l.geom().isNull() && r.geom().isNull()) {
                return l.id() < r.id();
            }
            if (l.geom().isNull()) {
                return true;
            }
            if (r.geom().isNull()) {
                return false;
            }
            return geom.distance(l.geom()) < geom.distance(r.geom());
        });
}

} // namespace

void
ObjectsQueryLasso::control()
{
    const auto categories = parseCategories(
        request_.categories,
        request_.substitutionPolicy);
    Geom geom(createGeomFromJsonStr(request_.geomJson));
    if (categories.empty() || geom.isNull()) {
        return;
    }

    if (request_.threshold) {
        geom = geom.createBuffer(*request_.threshold);
    }
    auto workView = BranchContextFacade::acquireWorkReadViewOnly(
        request_.branchId, request_.dbToken);
    views::ObjectsQuery objectsQuery;
    objectsQuery.addCondition(views::CategoriesCondition(*workView, categories));
    if (request_.indoorLevelId) {
        objectsQuery.addCondition(views::AllOfServiceAttributesCondition(
            *workView,
            {{srv_attrs::SRV_INDOOR_LEVEL_ID, std::to_string(*request_.indoorLevelId)}}));
    } else if (!categories.count(CATEGORY_INDOOR_LEVEL)) {
        objectsQuery.addCondition(views::NoneOfServiceAttributesCondition(
            *workView, {srv_attrs::SRV_INDOOR_LEVEL_ID}));
    }
    if (request_.predicate == GeomPredicate::CoveredBy) {
        objectsQuery.addCondition(views::CoveredByGeometryCondition(*workView, geom));
    } else {
        objectsQuery.addCondition(views::IntersectsGeometryCondition(*workView, geom));
    }
    if (request_.expressionId) {
        auto workCore = cfg()->poolCore().slaveTransaction(request_.dbToken);
        auto expression = filters::StoredExpression::load(*workCore, *request_.expressionId);
        objectsQuery.addCondition(views::GenericSqlCondition(
            " (" +
            expression.parsed().viewFilterClause(*workCore, views::ObjectsQuery::VIEW_TABLE_ALIAS + ".") +
            ") "));
    }
    bool hasSimpleGeomCats = false;
    bool hasContourObjectsCats = false;
    for (const auto& categoryId : categories) {
        hasSimpleGeomCats = hasSimpleGeomCats || isSimpleGeomCategory(categoryId);
        hasContourObjectsCats = hasContourObjectsCats || isContourObjectCategory(categoryId);
    }
    if (hasSimpleGeomCats) {
        objectsQuery.exec(*workView, request_.branchId, request_.limit).swap(result_->viewObjects);
    }
    if (hasContourObjectsCats) {
        objectsQuery.setMode(views::ObjectsQuery::Mode::ContourObjects);
        const auto fromContourObjects = enrichWithServiceAttrs(
            objectsQuery.exec(*workView, request_.branchId, request_.limit),
            *workView, request_.branchId);

        result_->viewObjects.insert(
            result_->viewObjects.end(),
            fromContourObjects.begin(),
            fromContourObjects.end());
    }
    sortViewObjects(result_->viewObjects, geom);
    if (request_.limit && result_->viewObjects.size() > request_.limit) {
        const auto dummy = result_->viewObjects[0];
        result_->viewObjects.resize(request_.limit, dummy);
    }
}

} // namespace wiki
} // namespace maps
