#include "objects_query_poi_conflicts.h"

#include "poi_conflicts_common.h"

#include "maps/wikimap/mapspro/services/editor/src/utils.h"
#include "maps/wikimap/mapspro/services/editor/src/branch_helpers.h"
#include "maps/wikimap/mapspro/services/editor/src/views/objects_query.h"
#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"
#include "maps/wikimap/mapspro/services/editor/src/srv_attrs/registry.h"

namespace maps::wiki {
namespace {
const std::string TASK_NAME = "ObjectsQueryPoiConflicts";
} // namespace

ObjectsQueryPoiConflicts::ObjectsQueryPoiConflicts(const Request& request)
    : controller::BaseController<ObjectsQueryPoiConflicts>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{
}

std::string
ObjectsQueryPoiConflicts::Request::dump() const
{
    std::stringstream ss;
    ss << " geom: " << geomJson;
    ss << " token: " << dbToken;
    ss << " branch: " << branchId;
    if (indoorLevelId) {
        ss << " indoor-level-id: " << *indoorLevelId;
    }
    if (objectId) {
        ss << " object-id: " << *objectId;
    }
    return ss.str();
}

std::string
ObjectsQueryPoiConflicts::printRequest() const
{
    return request_.dump();
}

namespace {
bool
userHasAccessToGeoproducts(TUid uid, Token token, const PoiConflictingCategories& categoriesHolder)
{
    auto workCore = cfg()->poolCore().slaveTransaction(token);
    return userHasAccessToGeoproducts(uid, categoriesHolder, *workCore);
}
} // namespace

const std::string&
ObjectsQueryPoiConflicts::taskName()
{
    return TASK_NAME;
}

void
ObjectsQueryPoiConflicts::control()
{
    static const PoiConflictingCategories poiCategories;
    static const poi_conflicts::PoiConflicts poiConflicts;
    Geom geom(createGeomFromJsonStr(request_.geomJson));
    WIKI_REQUIRE(!geom.isNull() && geom->getGeometryTypeId() == geos::geom::GEOS_POINT,
        ERR_BAD_REQUEST, "Point type geometry required, got: " << request_.geomJson);

    auto workView = BranchContextFacade::acquireWorkReadViewOnly(
        request_.branchId, request_.dbToken);

    views::ObjectsQuery objectsQuery;
    objectsQuery.addCondition(views::CategoriesCondition(*workView, poiCategories()));
    if (request_.indoorLevelId) {
        objectsQuery.addCondition(views::AllOfServiceAttributesCondition(
            *workView,
            {{srv_attrs::SRV_INDOOR_LEVEL_ID, std::to_string(*request_.indoorLevelId)}}));
    } else {
        objectsQuery.addCondition(views::NoneOfServiceAttributesCondition(
            *workView, {srv_attrs::SRV_INDOOR_LEVEL_ID}));
    }
    const auto searchGeom = geom.createBuffer(poiConflicts.maxConflictDistanceMercator());
    objectsQuery.addCondition(views::IntersectsGeometryCondition(*workView, searchGeom));
    auto viewObjects = objectsQuery.exec(*workView, request_.branchId);
    const auto escalateGeoproduct = userHasAccessToGeoproducts(request_.uid, request_.dbToken, poiCategories);
    for (const auto& viewObject : viewObjects) {
        if (request_.objectId && request_.objectId == viewObject.id()) {
            continue;
        }
        const auto zoomConflictKind = conflictKind(
            ObjectsQueryPoiConflicts::Request::IsGeoproductParam::True == request_.isGeoproduct,
            isGeoproduct(viewObject));
        const auto zoom = poiConflicts.conflictZoom(geom.distance(viewObject.geom()), zoomConflictKind);
        if (!zoom) {
            continue;
        }
        result_->viewObjectsConflicts[*zoom]
            [conflictSeverity(*zoom, escalateGeoproduct, zoomConflictKind)].push_back(viewObject);
    }
}

} // namespace mpas::wiki
